#!/usr/bin/python -u

"""
Takes in text data from a producer and inserts it
into a NilmDB stream. 

John Donnal 2016
"""

import argparse
import nilmdb.client
import nilmtools.filter
from nilmdb.utils.time import (timestamp_to_human,
                               seconds_to_timestamp,
                               parse_time,
                               now as time_now)
from auto_decimate import Decimator
import sys
import pdb
import numpy as np
import json
import yaml
import time
from db_file import DbFile
from db_stream import DbStream

#Timestamp formats
TS_MICROSECOND = 'us'
TS_MILLISECOND = 'ms'
TS_STRING = 'string'

class Consumer:
    def __init__(self, config):
        self.initialized = False
        self.config = config

        # initialize state variables:
        self.input_fifo = None
        self.buffer_size = None
        
    def initialize(self):
        
        self.client = nilmdb.client.numpyclient.\
                      NumpyClient("http://localhost/nilmdb")        
        # setup the input stream
        if(self.__setup_input()==False):
            return False
        self.dbFile = DbFile()
        # setup the destination stream
        if(self.dbFile.buildFromConfig(self.config)==False):
            return False
        # create the stream if necessary
        if(self.__create_stream() == False):
            return False # error in create stream

        self.initialized = True
        return True
    
    def __setup_input(self):
        try:
            self.input_fifo = self.config['fifo_path']
            self.buffer_size = int(self.__set('buffer_size',1))
            self.ts_format = self.__set('ts_format','us')
            if(self.ts_format != TS_MICROSECOND and
               self.ts_format != TS_MILLISECOND and
               self.ts_format != TS_STRING):
                print "[ERROR]: invalid ts_format [%s], "+\
                    "use 'us','ms', or 'string'"
                return False
            
        except KeyError as e:
            print "config file missing %s"%e[0]
            return False
        except ValueError as e:
            print "[ERROR]: buffer_size must be an integer"
            return False
        return True
    
    def __set(self,key, default):
        try:
            return self.config[key]
        except KeyError:
            return default

    def __create_stream(self):
        try:
            path = self.config['db_path']
        except KeyError as e:
            print "[ERROR]: missing db_path"
            return False
        # make sure the path is valid
        if(len(path[1:].split('/'))!=2):
            print "[ERROR]: invalid path %s, must be /group/file"%path
            return False
        group = path[1:].split('/')[0]
        group_path = "/%s/info"%group
        num_cols = self.dbFile.numCols()
        # 1. create the group if necessary
        info = nilmtools.filter.get_stream_info(self.client,group_path)
        if not info:
            print "creating group [%s]"%group
            self.client.stream_create(group_path,"uint8_1")
        # 2. check if the stream itself is present
        info = nilmtools.filter.get_stream_info(self.client,path)
        if info:
            # 2a: it exists, make sure the data type fits the configs
            print "checking for valid configs"
            if(info.layout_count != self.dbFile.numCols()):
                print("[ERROR]: config has %d fields, %s has %d"%\
                      (num_cols,path,info.layout_count))
                return False
        else:
            # 2b: it doesn't exist, make it
            print "Creating stream"
            config = self.dbFile.getConfig()
            config_key = {"config_key__":json.dumps(config)}
            self.client.stream_create(path,'float32_%d'%num_cols)
            self.client.stream_update_metadata(path,config_key)
        # 3. Build a numpy inserter context
        self.path = path
        self.decimator = Decimator(4,path,width=
                                   self.dbFile.numCols())
        return True
    def parse_ts(self,ts):
        if(self.ts_format==TS_MICROSECOND):
            return int(float(ts))
        elif(self.ts_format==TS_MILLISECOND):
            return 1000*int(float(ts))
        else: #self.ts_format==TS_STRING
            return parse_time(ts)

    def run(self):
        if self.initialized==False:
            print "error, call consumer.initialize() first"
            return
        
        #read timestamped data from source

        print("starting to read...")
        last_ts = 0
        start_ts = 0
        ts_array = np.empty((self.buffer_size,1),dtype=np.uint64)
        val_array = np.empty((self.buffer_size,
                              self.dbFile.numCols()),dtype=np.float32)
        while(True):
            try:
                with open(self.input_fifo,'r') as fifo:
                    # read a chunk of data of buffer_size
                    for i in range(self.buffer_size):
                        line = ''
                        while(line == ''):
                              line = fifo.readline()
                        vals = line.split(' ')
                        ts_array[i] = self.parse_ts(vals[0])
                        
                        val_array[i] = [float(x) for x in vals[1:len(vals)]]
                    # set up the interval
                    data = np.hstack((ts_array,val_array))
                    if(last_ts==0):
                        start_ts = int(data[0][0])
                    else:
                        start_ts = last_ts
                    last_ts = int(data[-1][0])+1
                    # insert the data into the database
                    with self.client.\
                         stream_insert_numpy_context(self.path,
                                                     start=start_ts, 
                                                     end=last_ts) as ctx: 
                        ctx.insert(data)
                    # decimate the data
                    self.decimator.process(data)
            except ValueError as e:
                time.sleep(0.2)
                pdb.set_trace()
                print e
                print "ValueError: %s"%e[0]
            except :
                print "Error processing input"
                time.sleep(2)
	time.sleep(5)
                
        
def main():
    parser = argparse.ArgumentParser( description = "Demo data producer")
    parser.add_argument("config_file", help="configuration file")
    args = parser.parse_args()
    # try to load the config file
    try:
        with open(args.config_file,'r') as f:
            config = yaml.load(f)
    except IOError:
        print "can't load configuration file at [%s], is it missing?"%args.config_file
        exit(1)
    # 1.) build the Consumer object
    consumer = Consumer(config)
    # 2.) initialize it and check for errors
    if(consumer.initialize() == False):
        print "error starting the consumer, exiting"
        exit(1)
    # 3.) run it, this shouldn't return
    consumer.run()

if __name__=="__main__":
    main()
