#!/usr/bin/python3 -u

from joule.utilities import time_now
from joule.client import ReaderModule
import asyncio
import numpy as np
import textwrap

import u3
import LabJackPython
import time
import traceback
import sys

ARGS_DESC = """
---
:name:
  LabJack U3 Reader
:author:
  John Donnal
:license:
  Open
:url:
  http://git.wattsworth.net/wattsworth/labjack_modules.git
:description:
  Read analog FIO/EIO channels on U3 LabJack
:usage:
  Read analog data from FIO and EIO channels on the U3 LabJack. The
  FIO channels are on screw terminals and the EIO channels are available 
  on the expansion board connector. On the HV model FIO0-3
  are marked AIN0-3 and accept voltages between +/- 10V. All other channels
  are 0-2.44V. *Specify at least 2 channels*

  | Arguments     | Description        |
  |---------------|--------------------|
  |``channels``   | comma separated channel numbers (0-15) |
  |``rate``       | data rate in Hz    |

:inputs:
  None

:outputs:
  output
  : float32 with N elements specified by [channels] argument

:stream_configs:
  #output#
    [Main]
    name = LabJack Data
    path = /path/to/output
    datatype = float32
    keep = 1w
    
    # [channels] number of elements
    # units are voltages
    [Element1]
    name = Channel 0
    units = V 

    [Element2]
    name = Channel 1
    units = V 

    [Element3]
    name = Channel 2
    units = V 
    
    #additional elements...

:module_config:
  [Main]
  name = LabJack U3 Reader
  exec_cmd = labjack_u3
  
  [Arguments]
  #valid values are [0-15]
  channels = 0,1,2 
  #data rate in Hz
  rate = 500

  [Outputs]
  output = /path/to/output
---
"""

class U3Reader(ReaderModule):
    "Read data from U3 LabJack"

        
    def custom_args(self, parser):
        grp = parser.add_argument_group("module",
                                        "module specific arguments")
        #--rate
        grp.add_argument("--rate", type=float, required=True,
                         help="sample rate in Hz")
        #--channels
        grp.add_argument("--channels", required=True)
        parser.description = textwrap.dedent(ARGS_DESC)

    async def run(self, parsed_args, output):
        channels = [int(x) for x in parsed_args.channels.split(',')]
        rate = parsed_args.rate
        data_ts_inc = 1e6 / rate
        self.stop_requested = False

        # set up timestamps
        data_ts = int(time.time()*1e6)
        # initialize the buffer
        self.buffer = [[] for x in range(len(channels))]
        try:
            device = self._setup_u3(channels, parsed_args.rate)
            device.streamStart()
            for chunk in device.streamData():
                if(self.stop_requested):
                    break
                if(chunk is None):
                    print ("LabJack did not return data, exiting")
                    break
                if(chunk['errors']!=0 or chunk['missed']!=0):
                    sys.stderr.write("errors: %d, missed: %d\n" % (
                        chunk['errors'],chunk['missed']))
                    
                (data_ts, block) = self.build_block(chunk,data_ts,
                                                    data_ts_inc,
                                                    channels)
                await output.write(block)
                await asyncio.sleep(.01)
            device.streamStop()
            device.close()
        except LabJackPython.LabJackException as e:
            print("LabJack Error: "+str(e))

    def build_block(self,r,data_ts,data_ts_inc,channels):
        # raw data may different sample counts per channel
        raw_data = [r['AIN%d'%c] for c in channels]
        # add in any buffered data
        data = [self.buffer[i]+raw_data[i] for i in range(len(channels))]
        # put extra data in the buffer
        shortest_row = min([len(d) for d in data])
        longest_row = max([len(d) for d in data])
        self.buffer = [d[shortest_row:longest_row] for d in data]
        #sys.stderr.write(repr(self.buffer)+"\n")
        
        # work with uniform data size
        data = [d[:shortest_row] for d in data]
        nrows = shortest_row
        block = np.empty((nrows,len(channels)+1))
        top_ts = data_ts + nrows * data_ts_inc
        ts = np.array(np.linspace(data_ts, top_ts,
                                  nrows, endpoint=False), dtype=np.uint64)
        block = np.vstack((ts,*data)).T
        return (top_ts, block)

    def _setup_u3(self, channels, rate):

        device = u3.U3()
        try:
            device.streamStop()
            sys.stderr.write("warning: stopped a previous stream\n")
        except:
            pass
        device.configU3()
        
        # set all IO channels to analog signals
        device.configIO(FIOAnalog=0xFF, EIOAnalog=0xFF)
            
        #single ended measurements
        gnd_channels = [31 for x in channels]
        device.streamConfig(NumChannels=len(channels),
                            PChannels=channels,
                            NChannels=gnd_channels,
                            ScanFrequency=rate,
                            SamplesPerPacket=25)
        return device

def main():
    r = U3Reader()
    r.start()
    
if __name__ == "__main__":
    main()
