#!/usr/bin/python

"""
Finalize calibration data
-create the recovery matrix for the current
-store all calibration data in appropriate files
"""

import numpy as np
import numpy.linalg as linalg
import yaml
import pdb
import argparse
import os
from scipy import ndimage
from matplotlib import pyplot as plt

CONFIG_DIR = "/opt/configs/"
METER_FILE = CONFIG_DIR+"meters.yml"
METER_CAL = CONFIG_DIR+"meters"
CALIB_DATA   = CONFIG_DIR+"/cal_data"

def run(meter_config,debug=False):
        
    cal_config = meter_config["calibration"]
        
    #create the calibration fit matrix
    fit_matrix=linalg.pinv(cal_config["sensor_matrix"])
    cal_config["current_matrix"]=fit_matrix
    sinefit_rotations = cal_config["sinefit_rotations"]
    #if this is a no neutral calibration
    if(not cal_config["has_neutral"]):
        ll_rotations = np.array(sinefit_rotations) #line to line rotations
        #find angle between voltage vectors
        phase_angle = -1*(sinefit_rotations[1]-sinefit_rotations[0])
        phase_angle%=2*np.pi
        #if the phase angle is 120/240deg then we have AC and CB legs
        #we want an angle of    60/300deg which would be AC and BC legs
        if(np.cos(phase_angle)<0):
            #we need to flip the CB leg
            print("CB sense on 2nd leg, flipping to BC sense")
            fit_matrix[1]*=-1
            sinefit_rotations[1]+=np.pi
            sinefit_rotations%=2*np.pi #keep angle 0<-->2pi
        else:
            fit_matrix[1]*=1
        #find angle between voltage vectors
        phase_angle = -1*(sinefit_rotations[1]-sinefit_rotations[0])
        phase_angle%=2*np.pi
        #if angle is 60 deg then cal was performed AC -> BC
        #if angle is 300 deg then cal was performed BC -> AC
        #   flip inputs around so the cal looks AC -> BC
        if(np.sin(phase_angle)<0):
            print("BC -> AC cal, flipping")
            cal_config["clockwise"]=False
            tmp=np.array(fit_matrix[0])
            fit_matrix[0]=np.array(fit_matrix[1])
            fit_matrix[1]=tmp
            tmp = sinefit_rotations[0]
            sinefit_rotations[0] = sinefit_rotations[1]
            sinefit_rotations[1] = tmp
        else:
            cal_config["clockwise"]=True
            
        #Now rotate the voltage vectors so they are line to
        #neutral instead of line to line, this means subtract
        #pi/6 off AC and  each phase (add to sinefit rotation)
        sinefit_rotations[0]+=np.pi/6
        sinefit_rotations[1]-=(np.pi/6)
        #Create the phase C leg and put everything into full_rotations
        full_rotations = [float(sinefit_rotations[0]),
                          float(sinefit_rotations[1]),   
                          float(sinefit_rotations[0]+2*np.pi/3)]
        full_rotations[2]%=2*np.pi
        full_matrix = np.array([[1,0],[0,1],[-1,-1]])
        cal_config["full_current_matrix"] = full_matrix.dot(fit_matrix)
        cal_config["full_sinefit_rotations"] = full_rotations
    #write the new config values
    for key in cal_config:
        if(type(cal_config[key])==np.ndarray):
            cal_config[key]=cal_config[key].tolist()
    yaml_data = yaml.safe_dump(cal_config, default_flow_style=False, canonical=False)
    cal_file = METER_CAL+"/"+meter_config["name"]+".yml"
    with open(cal_file,'w') as f:
        f.write(yaml_data)

    #----Debug Plotting Code---

    if(debug):
#        plt.rc('text', usetex=True)
#        plt.rc('font', family='serif')
        fig = plt.figure(facecolor='white')
        orig_plot = fig.add_subplot('111',aspect='equal')
        fig2 = plt.figure(facecolor='white')
        rot_plot = fig2.add_subplot('111', aspect='equal')
        phase_names=['A','B','C']
        #----rotation function----
        def rot(x,y):
            d=sinefit_rotations[0]
            M=np.array([[np.cos(d),-1*np.sin(d)],[np.sin(d),np.cos(d)]])
            (rx,ry)=M.dot([x,y])
            return (rx,ry)
        
        #----add sinefit voltage basis---
        if(cal_config["has_neutral"]):
            rotations = sinefit_rotations
        else:
            rotations = full_rotations
        for i in range(len(rotations)):
            x = np.cos(rotations[i])
            y = np.sin(rotations[i])
            (rx,ry)=rot(x,y)
            
            orig_plot.annotate("",xy=(x,y), xytext=(0,0),
                         arrowprops=dict(arrowstyle='simple',facecolor="grey",
                                         alpha=0.5))
            orig_plot.text(x,y,phase_names[i])
            rot_plot.annotate("",xy=(rx,ry), xytext=(0,0),
                         arrowprops=dict(arrowstyle='simple',facecolor="grey",
                                         alpha=0.5))
            rot_plot.text(rx,ry,phase_names[i])

        #----if no neutral add line to line cal vectors--
        if(not cal_config["has_neutral"] and False):
            if(cal_config["clockwise"]):
                labels=['AC','BC']
            else:
                labels=['BC','AC']
            for i in range(2):
                x = np.cos(-1*ll_rotations[i])
                y = np.sin(-1*ll_rotations[i])
                (rx,ry)=rot(x,y)

                orig_plot.annotate("",xy=(x,y), xytext=(0,0),
                             arrowprops=dict(arrowstyle='simple',
                                             facecolor="grey",
                                             alpha=0.5))
                orig_plot.text(x,y,labels[i])
                rot_plot.annotate("",xy=(rx,ry), xytext=(0,0),
                             arrowprops=dict(arrowstyle='simple',
                                             facecolor="grey",
                                             alpha=0.5))
                rot_plot.text(rx,ry,labels[i])

        #----add sensor pq vectors----
        pq_coeffs = np.array(cal_config["pq_coeffs"],dtype=float)
        colors = ['b','r','k']
        for i in range(len(pq_coeffs)):
            s_coeffs = pq_coeffs[i]
            for j in range(len(s_coeffs)):
                if(j==0):
                    if(not(cal_config["has_neutral"])):
                        if(cal_config["clockwise"]):
                            phase_names = ['A-C','B-C']
                        else:
                            phase_names = ['B-C','A-C']    
                        label = "Cal %s"%(phase_names[i]) #add label for first sensor
                    else:
                        label = "Cal Phase %s"%(phase_names[i])
                else:
                    label = None
                x = s_coeffs[j,0]; y = s_coeffs[j,1]
                (rx,ry)=rot(x,y)
                
                orig_plot.plot([0,s_coeffs[j,0]],[0,s_coeffs[j,1]],
                         label=label,color=colors[i])
                
                rot_plot.plot([0,rx],[0,ry],
                         label=label,color=colors[i])

        #----add voltage sensor reference vector---
        x = 1; y = 0
        (rx,ry)=rot(x,y)
        orig_plot.plot([0,1],[0,0],'k--',label="E-sensor")
        rot_plot.plot([0,rx],[0,ry],'k--',label="E-sensor")

        #---add a unit circle for reference
        circle = plt.Circle((0,0),1,fill=False)
        orig_plot.add_artist(circle)
        circle = plt.Circle((0,0),1,fill=False)
        rot_plot.add_artist(circle)

        #---set up the plot area (axis,title,size,etc)
        orig_plot.set_ylim([-1.2,1.2])
        orig_plot.set_xlim([-1.2,1.2])
        orig_plot.axis("off")
        rot_plot.set_ylim([-1.2,1.2])
        rot_plot.set_xlim([-1.2,1.2])
        rot_plot.axis("off")
        rot_plot.set_title("Complete Phase Reconstruction")
        orig_plot.set_title("Phases referenced to E Sensor")
        rot_plot.legend(bbox_to_anchor=(0.,0,1,0), loc=3,
                   ncol=3, mode="expand", borderaxespad=0.)
        orig_plot.legend(bbox_to_anchor=(0.,0,1,0), loc=3,
                   ncol=3, mode="expand", borderaxespad=0.)

        #---display and wait for user to acknowledge
        plt.ion()
        plt.show()
        #---save the final phase plot in the CAL_DATA dir
        fig2.savefig(CALIB_DATA+"/%s.png"%meter_config["name"],bbox_inches='tight',dpi=300)
        x = input("Press [enter] to continue...")


