"""
Reads data from usb or ethstream device
"""
import signal
import asyncio

from nilm.meter import meter
from nilm.meter.utils import MeterConfigurationError
import logging
from .serialcapture import SerialCapture
from .ethstreamcapture import EthstreamCapture
from .errors import CaptureError
from cliff.command import Command
from joule.client import ReaderModule
import sys


class CaptureCmd(Command):

    def __init__(self, app, app_args, cmd_name=None):
        super(CaptureCmd, self).__init__(app, app_args, cmd_name)
        self.fd = None  # nothing to close on exit

    def get_parser(self, prog_name):
        parser = super(CaptureCmd, self).get_parser(prog_name)
        parser.add_argument("meter",
                            help='name from meters.yml (eg meter1)')
        # --config-file = /opt/configs/meters.yml
        parser.add_argument("--config-file", "-f",
                            default="/opt/configs/meters.yml",
                            dest="config_file",
                            help='file')
        # --calibration-directory = /opt/configs/meters
        parser.add_argument("--calibration-directory", "-d",
                            default="/opt/configs/meters",
                            dest="cal_dir",
                            help='directory')
        # --joule: flag to run as joule module
        joule.utils.client.add_args(parser)
        return parser

    def take_action(self, parsed_args):
        reader = readers.Capture()
        reader.start(argv=
        try:
            meters = meter.load_meters(parsed_args.config_file,
                                       parsed_args.cal_dir)
        except MeterConfigurationError as e:
            logging.error("Error, invalid meter configurations")
            logging.error(e)
            return 1
        my_meter = meters[parsed_args.meter]
        if(my_meter['type'] == 'noncontact'):
            my_capture = SerialCapture(my_meter['tty'])
        else:
            my_capture = EthstreamCapture(my_meter['ip_address'])

        (pipes_in, pipes_out) = joule.utils.client.build_pipes(parsed_args)

        if(len(pipes_out) == 0):
            output_func = self._ascii_output_factory()
        else:
            output_func = self._joule_output_factory(pipes_out['sensor'])

        try:
            loop = asyncio.get_event_loop()
            loop.set_debug(False)
            loop.add_signal_handler(signal.SIGINT, my_capture.stop)
            loop.add_signal_handler(signal.SIGTERM, my_capture.stop)
            loop.run_until_complete(my_capture.run(output_func))
            loop.close()
        except CaptureError as e:
            print("Error!")
            print(e)
        finally:
            my_capture.close()
            self._close_fd()

    def _ascii_output_factory(self):
        async def output(np_data):
            for row in np_data:
                print(" ".join(["%d" % x for x in row]))
        return output

    def _joule_output_factory(self, pipe):
        async def output(np_data):
            await pipe.write(np_data)
        return output

    def _close_fd(self):
        if(self.fd is not None):
            self.fd.close()
