import subprocess  # for pinging LabJack meters
from nilm.meter.utils import Status, print_error, print_warning
import os  # for /dev/null


#------------ Connectivity Check --------------

# Check if the meters are connected and unique
#  Error for invalid configs
#  Warning for meters that cannot be reached
def check_meter_connections(meters):
    my_status = Status.ok
    # 1 make sure IP addresses and serial numbers are unique
    ip_addrs = []
    serials = []
    for name in meters:
        meter = meters[name]
        if(meter["type"] == "contact"):
            ip_addrs.append(meter["ip_address"])
        elif(meter["type"] == "noncontact"):
            serials.append(meter["serial_number"])
        else:
            print_error(
                "%s, invalid [type] must be [contact|noncontact]" % name)
            return Status.error
    if(len(set(ip_addrs)) != len(ip_addrs)):
        print_error("Each contact meter must have a unique IP address")
        return Status.error
    if(len(set(serials)) != len(serials)):
        print_error("Each noncontact meter must have a unique serial number")
        return Status.error
    # 2 check if the meter is reachable
    for name in meters:
        meter = meters[name]
        if(meter["enabled"] == False):
            continue
        if(meter["type"] == "contact"):
            my_status = min(my_status, check_connectivity_contact(
                name, meter["ip_address"]))
        else:
            my_status = min(
                my_status, check_connectivity_noncontact(name, meter))
    return my_status

# Check connectivity of LabJack meter
#  try to ping it 2x with a timeout of 1 second


def check_connectivity_contact(name, ip_address):
    r = subprocess.call(["ping", "-c", "2", "-W", "1", ip_address], stdout=subprocess.DEVNULL,
                        stderr=subprocess.DEVNULL)
    if(r != 0):
        print_warning(
            "%s cannot reach meter at IP address [%s]" % (name, ip_address))
        return Status.warn
    return Status.ok

# Check connectivity of USB meter
#  see if it has enumerated correctly


def check_connectivity_noncontact(name, meter):
    if(not os.path.exists(meter['tty'])):
        print_warning("%s device [%s] is not connected" % (name, meter['tty']))
        return Status.warn
    return Status.ok
