"""
Given a meter, show the realtime values of
the sensors.

"""

import asyncio
import argparse
import os
import subprocess
import time
import numpy as np
import threading
import warnings

import matplotlib
from matplotlib import pyplot as plt

from cliff.command import Command

from joule.utils.localnumpypipe import LocalNumpyPipe
from nilm.readers import capture
from nilm.meter import meter
from nilm.meter.utils import MeterConfigurationError


METERS_FILE = "/opt/configs/meters.yml"
METER_CAL = "/opt/configs/meters"
CALIB_DATA_DIR = "/opt/configs/cal_data"
TYPE_NON_CONTACT = "noncontact"

NON_CONTACT_NUM_POINTS = 300
CONTACT_NUM_POINTS = 600

ESC_SEQ = "\x1b["
COL_RESET = ESC_SEQ + "0m"
COL_RED = ESC_SEQ + "6;30;41m"
COL_GREEN = ESC_SEQ + "6;30;42m"
COL_YELLOW = ESC_SEQ + "6;30;43m"
COL_BLUE = ESC_SEQ + "34;01m"
COL_MAGENTA = ESC_SEQ + "35;01m"
COL_CYAN = ESC_SEQ + "36;01m"

FNULL = open(os.devnull, 'w')

PLOT_WIDTH = 200


class ScopeCmd(Command):
    "Realtime scope for contact and non-contact meters"
    
    def get_parser(self, prog_name):
        parser = super(ScopeCmd, self).get_parser(prog_name)
        # meter
        parser.add_argument('meter', help="meterXXXX")
        # --channels
        parser.add_argument('--channels', '-c', nargs='+',
                            help="Channels to plot", required=True)
        # --config-file = /opt/configs/meters.yml
        parser.add_argument("--config-file", "-f",
                            default=METERS_FILE,
                            dest="config_file",
                            help='file')
        return parser

    def take_action(self, parsed_args):

        channels = [int(x) for x in parsed_args.channels]
        try:
            meters = meter.load_meters(parsed_args.config_file)
            my_meter = meters[parsed_args.meter]
        except MeterConfigurationError as e:
            print(("[" + COL_RED + "ERROR" + COL_RESET + "]: %s" % e))
            exit(1)
        except KeyError as e:
            print("meter [%s] not found in meters.yml" % e)
            exit(1)

        ##########
        # Stop the nilm-capture service if it is running
        ret = subprocess.call(["pgrep", "jouled"],
                              stdout=FNULL, stderr=FNULL)
        if(ret == 0):
            subprocess.call(["sudo", "service", "jouled", "stop"],
                            stdout=FNULL, stderr=FNULL)
            time.sleep(5)
            print("waiting 5 seconds for data capture to stop")
            restart_capture = True
        else:
            restart_capture = False

        # pipes to receive data
        npipe = LocalNumpyPipe("raw", layout="int16_8")

        # capture reader
        capture_args = argparse.Namespace(
            meter=my_meter['name'],
            max_gap=1,
            disable_alignment=True)

        my_capture = capture.Capture()

        # setup the plot
        self.plot_width = CONTACT_NUM_POINTS
        if(my_meter['type'] == TYPE_NON_CONTACT):
            self.plot_width = NON_CONTACT_NUM_POINTS

        # suppress the rcParam warning 
        warnings.filterwarnings("ignore",category=UserWarning,module='matplotlib')
        (plot, fig, data, ax, bkgd) = self.setup_plot(my_meter, channels)
        fig.canvas.mpl_connect('close_event', self.shutdown)
        # setup the tasks and run the loop
        tasks = [
            asyncio.ensure_future(
                my_capture.run(capture_args, npipe, meter=my_meter)),
            asyncio.ensure_future(self.update_plot(
                npipe, channels, fig, data, ax, bkgd))]
        loop = asyncio.get_event_loop()
        self.shutdown = False
        threading.Thread(target=self.run_loop, args=(loop, tasks)).start()
        plot.show()
        self.shutdown = True
        my_capture.stop()
        time.sleep(1)
        #force close 
        loop.close()

    def run_loop(self, loop, tasks):
        asyncio.set_event_loop(loop)
        loop.run_until_complete(asyncio.gather(*tasks))
        
    def shutdown(self):
        print("shutting down")

    async def update_plot(self, npipe, channels, fig, data, ax, bkgd):
        while(not self.shutdown):
            sensor_data = await npipe.read(flatten=True)
            if(len(sensor_data) < self.plot_width):
                continue  # not enough data to plot
            sensor_data = sensor_data - np.mean(sensor_data, axis=0)
 #           fig.canvas.restore_region(bkgd)
            for i in range(len(channels)):
                points = data[i]
                idx = channels[i]+1  # ts is column 0
                cur_data = points.get_data()
                points.set_data(cur_data[0],
                                sensor_data[:self.plot_width, idx])
#                ax.draw_artist(points)
#            fig.canvas.blit(ax.bbox)
            fig.canvas.draw()
            consumed_data = len(sensor_data)-len(sensor_data) % self.plot_width
            npipe.consume(consumed_data)
            await asyncio.sleep(.1)

    def setup_plot(self, my_meter, channels):

        # remove navigation toolbar
        matplotlib.rcParams['toolbar'] = 'None'

        fig, ax = plt.subplots(1, 1)
        ax.hold(True)
        ax.grid(True)

        if(my_meter["type"] == TYPE_NON_CONTACT):
            ax.set_ylim([-1000, 1000])
        else:
            ax.set_ylim([-6000, 6000])
        ax.set_xlim([0, self.plot_width])
        ax.set_autoscale_on(False)
        ax.set_xticks([])
        background = fig.canvas.copy_from_bbox(ax.bbox)
        plot_data = []
        if(my_meter["type"] == TYPE_NON_CONTACT):
            ax.set_title("Raw data from non-contact meter [%s]" %
                         my_meter["serial_number"])
            for idx in channels:
                if idx < 0 or idx > 7:
                    print("Error, invalid channel: %d" % idx)
                    exit(1)
                # figure out this channel's role
                role = "[%d] --unused--" % idx
                if(idx == my_meter["sensors"]["voltage"]["sensor_index"]):
                    role = "[%d] Voltage" % idx
                else:
                    for k in my_meter["sensors"]["current"]["sensor_indices"]:
                        if idx == k:
                            role = "[%d] Current" % idx
                data = np.zeros(self.plot_width)
                points, = ax.plot(
                    list(range(self.plot_width)), data, label=role)
                plot_data.append(points)

        else:
            ax.set_title(
                "Raw data from contact meter [%s]" % my_meter["ip_address"])
            for idx in channels:
                if(idx < 0 or idx > 5):
                    print("Error, invalid channel: %d" % idx)
                    exit(1)
                if(idx < 3):
                    role = "[%d] Current" % idx
                else:
                    role = "[%d] Voltage" % idx
                data = np.zeros(self.plot_width)
                points, = ax.plot(
                    list(range(self.plot_width)), data, label=role)
                plot_data.append(points)
        ax.legend()
        plt.draw()
        return (plt, fig, plot_data, ax, background)
