"""
Reads data from usb or ethstream device
"""
import logging


from joule.client import ReaderModule
from nilm.meter.meter import load_meters
from nilm.meter.utils import MeterConfigurationError
from .serialcapture import SerialCapture
from .ethstreamcapture import EthstreamCapture
from .errors import CaptureError


class Capture(ReaderModule):

    def __init__(self):
        super(Capture, self).__init__("NILM Data Capture")
        self.description = "NILM data acquisition"
        self.help = """ read data from contact and non-contact meters"""
    
    def custom_args(self, parser):
        parser.add_argument("meter",
                            help='name from meters.yml (eg meter1)')
        # --config-file = /opt/configs/meters.yml
        parser.add_argument("--config-file", "-f",
                            default="/opt/configs/meters.yml",
                            dest="config_file",
                            help='file')
        # --calibration-directory = /opt/configs/meters
        parser.add_argument("--calibration-directory", "-d",
                            default="/opt/configs/meters",
                            dest="cal_dir",
                            help='directory')
        # --max-gap = 10
        parser.add_argument("--max-gap", "-g",
                            default="10", type=int,
                            dest="max_gap",
                            help="max misalignment between sample timestamp and system clock (in seconds)")
        # --disable-clock-alignment = False
        parser.add_argument("--disable-clock-alignment", "-x",
                            default=False, type=bool,
                            dest="disable_alignment",
                            help="do not align timestamps to system clock")
        
    async def run(self, parsed_args, output, meter=None, nrows=0):

        try:
            if(meter==None):
                meters = load_meters(parsed_args.config_file,
                                           parsed_args.cal_dir)
                meter = meters[parsed_args.meter]
        except MeterConfigurationError as e:
            logging.error("Error, invalid meter configurations")
            logging.error(e)
            return 1
        if(meter['type'] == 'noncontact'):
            self.my_capture = SerialCapture(meter['tty'])
        else:
            self.my_capture = EthstreamCapture(meter['ip_address'])
        try:
            await self.my_capture.run(output,
                                      nrows=nrows,
                                      max_gap=parsed_args.max_gap,
                                      disable_alignment=parsed_args.disable_alignment)
        except CaptureError as e:
            print("Error!")
            print(e)
        finally:
            self.my_capture.close()

    def stop(self):
        self.my_capture.stop()

def main():
    r = Capture()
    r.start()
    
if __name__ == "__main__":
    main()
