
import logging
import json
import asyncio
import numpy as np
import argparse

from nilm.meter.utils import MeterConfigurationError
from nilm.meter import meter

from joule.utils.localnumpypipe import LocalNumpyPipe
from joule.client import FilterModule

from .reconstructor import Reconstructor
from .sinefit import Sinefit
from .prep import Prep

ARGS_DESC = """
Operation
--------------
process full filter stack:

                            ,--iv       ,--sinefit
                            |           |
       raw-->[reconstructor]-->[sinefit]-->[prep]--> prep (A/B/C)
                             \_______________/

Pipes
--------------
Inputs:

  'raw' float32 <ch1, ch2, ... chM> for M ADC channels

Outputs (eg for 3 phase contact merged):

    'iv'             float32 <V_A, I_A, V_B, I_B> eg for 2 phase
    'zero_crossings' float32 <freq, amp, phase>
    'prep'           float32 <P1A, P1B, P1C, ... Q7A, Q7B, Q7C>

Arguments
------------------
  see below
"""


class RawToPrep(FilterModule):

    def __init__(self):
        super(RawToPrep, self).__init__("RawToPrep")
        self.description = "run complete NILM toolchain"
        self.help = '''\
          convert raw -> iv -> zero_crossings -> prep
        '''
        self.arg_description = ARGS_DESC

    def custom_args(self, parser):
        parser.add_argument("meter",
                            help='name from meters.yml (eg meter1)')
        # --config-file = /opt/configs/meters.yml
        parser.add_argument("--config-file", "-f",
                            default="/opt/configs/meters.yml",
                            help='file')
        # --calibration-file = <override>
        parser.add_argument("--calibration-file", "-c")
        # --calibration-directory = /opt/configs/meters
        parser.add_argument("--calibration-directory", "-d",
                            default="/opt/configs/meters",
                            dest="cal_dir",
                            help='directory')
        # --merge-prep = False
        parser.add_argument("--merge-prep",
                            action="store_true",
                            dest="merge_prep",
                            help="store prep in a single stream")
        # --polar = False
        parser.add_argument("--polar",
                            action="store_true",
                            dest="polar",
                            help="compute mag,phase instead of PQ")

    async def run(self, parsed_args, inputs, outputs):

        # configure logger
        logger = logging.getLogger()
        logger.setLevel(logging.WARN)
        try:
            meters = meter.load_meters(parsed_args.config_file,
                                       parsed_args.cal_dir)
        except MeterConfigurationError as e:
            logging.error("Error, invalid meter configurations")
            logging.error(e)
            exit(1)

        my_meter = meters[parsed_args.meter]

        configs = {
            'reconstructor':  self._build_recon_configs(my_meter),
            'sinefit':        self._build_sinefit_configs(my_meter),
            'prep':           self._build_prep_configs(my_meter,
                                                       parsed_args.merge_prep,
                                                       parsed_args.polar)
        }

        tasks = self._run_filters_as_tasks(
            inputs, outputs, configs)

        for task in tasks:
            await task  # should never exit
            
#        loop = asyncio.get_event_loop()
#        loop.set_debug(False)
#        loop.run_until_complete(asyncio.gather(*tasks))

    def _run_filters_as_tasks(self, pipes_in, pipes_out, config):
        tasks = []
        
        out_iv = pipes_out['iv']
        # reconstructor --> sinefit
        r2s_iv = LocalNumpyPipe("r2s_iv", out_iv.layout)
        # reconstructor --> prep
        r2p_iv = LocalNumpyPipe("r2p_iv", out_iv.layout)
        r2s_iv.subscribe(out_iv)
        r2s_iv.subscribe(r2p_iv)

        out_zc = pipes_out['zero_crossings']
        # sinefit --> prep
        s2p_zc = LocalNumpyPipe("s2p_zc", out_zc.layout)
        s2p_zc.subscribe(out_zc)

        my_recon = Reconstructor()
        args = argparse.Namespace(config=json.dumps(config['reconstructor']))
        tasks.append(
            asyncio.ensure_future(
                my_recon.run(args,
                             {'raw': pipes_in['raw']},
                             {'iv': r2s_iv})))

        my_sinefit = Sinefit()
        args = argparse.Namespace(config=json.dumps(config['sinefit']))
        tasks.append(
            asyncio.ensure_future(
                my_sinefit.run(args,
                               {'iv': r2s_iv},
                               {'zero_crossings': s2p_zc})))

        # --- prep setup: streams may be separate or merged ---
        prep_pipes_out = {}
        for name in pipes_out:
            if('prep' in name):
                prep_pipes_out[name] = pipes_out[name]

        my_prep = Prep()
        args = argparse.Namespace(config=json.dumps(config['prep']))
        tasks.append(
            asyncio.ensure_future(
                my_prep.run(args,
                            {'iv': r2p_iv,
                             'zero_crossings': s2p_zc},
                            prep_pipes_out)))

        return tasks

    def _build_recon_configs(self, meter):

        def _compute_e_indices(meter):
            if(meter["type"] == "noncontact"):
                return [meter['sensors']['voltage']['sensor_index']]
            else:  # contact meter
                return meter['sensors']['voltage']['sensor_indices']

        def _compute_voltage_matrix(meter):
            if(meter["type"] == "noncontact"):
                return [meter['calibration']['voltage_scale']]
            else:  # contact meter
                voltage_scales = meter['sensors']['voltage']['sensor_scales']
                if(not(type(voltage_scales) is list)):
                    voltage_scales = [voltage_scales]*meter['phases']
                return np.diag(voltage_scales).tolist()
                
        def _compute_current_matrix(meter):
            if(meter["type"] == "noncontact"):
                has_neutral = meter['calibration']['has_neutral']
                if(not has_neutral):
                    return meter['calibration']['full_current_matrix']
                else:
                    return meter['calibration']['current_matrix']
            else:  # contact meter
                current_scales = meter['sensors']['current']['sensor_scales']
                if(not(type(current_scales) is list)):
                    current_scales = [current_scales]*meter['phases']
                return np.diag(current_scales).tolist()

        def _compute_integration_setting(meter):
            if(config['noncontact']):
                return meter['sensors']['voltage']['digitally_integrate']
            else:  # contact meter
                return False

        def _compute_sampling_freq(meter):
            if(config['noncontact']):
                return 3000
            else:  # contact meter
                return 8000
            
        config = {}
        config['noncontact'] = (meter["type"] == "noncontact")
        config['m_indices'] = meter['sensors']['current']['sensor_indices']
        config['e_indices'] = _compute_e_indices(meter)
        config['voltage_matrix'] = _compute_voltage_matrix(meter)
        config['current_matrix'] = _compute_current_matrix(meter)
        config['nominal_frequency'] = 60
        config['sampling_frequency'] = _compute_sampling_freq(meter)
        config['max_gap'] = 5  # seconds
        config['integrate'] = _compute_integration_setting(meter)
        return config

    def _build_sinefit_configs(self, meter):
        config = {}
        config['v_index']=1 #V_A for contact or 'V' for noncontact
        config['min_amp'] = 10
        config['min_freq'] = 40
        config['frequency'] = 60
        config['max_freq'] = 70
        return config

    def _build_prep_configs(self, meter, merge_output, polar):

        def _compute_rotations(meter):
            if(meter['type'] == 'noncontact'):
                if(meter['calibration']['has_neutral']):
                    return meter['calibration']['sinefit_rotations']
                else:
                    return meter['calibration']['full_sinefit_rotations']
            else:  # contact meter
                rotations = meter['sensors']['current']['sinefit_rotations']
                # convert to radians
                return [x*(2*np.pi)/360.0 for x in rotations]

        def _compute_current_indices(meter):
            if(meter['type'] == 'noncontact'):
                # [ts, V, --> I1, I2, I3]
                return np.arange(2, 2 + meter['phases']).tolist()
            else:  # contact meter
                # [ts, V1, V2, V3, --> I1, I2, I3]
                return np.arange(1 + meter['phases'], 1 + 2 * meter['phases']).tolist()

        def _compute_scale_factor(meter):
            line_rms = meter['sensors']['voltage']['nominal_rms_voltage']
            return line_rms / np.sqrt(2)  # convert amps to rms
            
        config = {}
        config['nshift'] = 1
        config['nharm'] = 4
        config['current_indices'] = _compute_current_indices(meter)
        config['rotations'] = _compute_rotations(meter)
        config['scale_factor'] = _compute_scale_factor(meter)
        config['merge'] = merge_output
        config['polar'] = polar
        return config

    
def main():
    filter = RawToPrep()
    filter.start()

    
if __name__ == "__main__":
    main()


