#------------- Disk Usage Check ------------------------------

# estimate required size given current settings
# warn if > 85% will be used
#

from joule.utils.nilmdb import Client
from nilm.meter.utils import print_error, print_warning, Status
import re

DISK_USAGE_THRESHOLD = 0.85


def check_disk_usage(settings, verbose=False):
    total_size = 0
    for name in settings:
        meter = settings[name]
        if(meter["enabled"] == False):
            continue
        stream_config = meter["streams"]
        # extract sinefit requirement
        sinefit = stream_config["sinefit"]
        if(sinefit["keep"] != False):
            x = calc_num_samples(sinefit["keep"], 60, sinefit["decimate"])
            total_size += x * (8 + 4 * 3)
        # extract prep requirement
        prep = stream_config["prep"]
        x = calc_num_samples(prep["keep"], 60, prep["decimate"])
        total_size += x * (8 + 4 * 9) * meter["phases"]
        # extract the iv requirement
        iv = stream_config["iv"]
        if(iv["keep"] != False):
            y = meter["phases"]
            if(meter["type"] == "contact"):
                rate = 8000
                y *= 2  # keep all voltages
            else:  # noncontact
                rate = 3000
                y += 1  # keep 1 voltage
            x = calc_num_samples(iv["keep"], rate, iv["decimate"])
            total_size += x * (8 + y * 9)
        # extract the sensor requirement
        if(meter["type"] == "noncontact"):
            sensor = stream_config["sensor"]
            if(sensor["keep"] != False):
                x = calc_num_samples(sensor["keep"], 3000, sensor["decimate"])
                total_size += x * (8 + 8 * 9)
    # compare this with the amount of space available
    client = Client("http://localhost/nilmdb")
    dbinfo = client.dbinfo()
    available_size = dbinfo["size"] + dbinfo["free"]
    percent = float(total_size) / float(available_size)
    if(percent > 1.0):
        print_warning("This configuration will require more disk space than available" +
                      "\n\t %0.2f GiB available and %0.2f GiB required" %
                      (float(available_size) / 2**30, float(total_size) / 2**30))
        return Status.warn
    elif(percent > DISK_USAGE_THRESHOLD):
        print_warning("This configuration will use %d%% of the disk" %
                      int(percent * 100.0))
        return Status.warn
    if(verbose):
        print("This configuration will require ~%0.2f GiB" % (float(total_size) / 2**30) +
              "\nThis is %d%% of the available space on the disk" % int(percent * 100.0))
    return Status.ok

# calculate number of rows saved by the specified config
#


def calc_num_samples(config, rate, decimated):
    if(config is False):
        return 0
    if(config.lower() == "none"):
        return 0
    x = re.match("(\d+)([hdwmy])", config)
    if(x == None):
        print_error("invalid [keep] configuration [%s]" % (config))
        return Status.error
    unit = x.group(2)
    val = int(x.group(1))
    if(unit == 'h'):
        multiple = 60 * 60
    elif(unit == 'd'):
        multiple = 60 * 60 * 24
    elif(unit == 'w'):
        multiple = 60 * 60 * 24 * 7
    elif(unit == 'm'):
        multiple = 60 * 60 * 24 * 7 * 30
    elif(unit == 'y'):
        multiple = 60 * 60 * 24 * 7 * 365
    if(decimated):
        multiple *= 2
    return rate * val * multiple
