import asyncio
import numpy as np
from .errors import CaptureError
from joule.utils.time import (timestamp_to_human,
                              seconds_to_timestamp, now as time_now)
import logging
import sys

ROW_BYTES = 6 * 2   # 6 channels, 2 byte ints
ROWS_PER_BLOCK = 1000  # read ~8 blocks per second
BLOCK_SIZE = ROW_BYTES * ROWS_PER_BLOCK


class EthstreamCapture():

    def __init__(self, ip_address):
        # tunable constants
        self.data_ts_inc = 1e6 / 8000.0  # 8KHz sampling

        self.ip_address = ip_address
        self.stop_requested = False

    async def run(self, output,
                  max_gap=10,
                  disable_alignment=False,
                  nrows=0):
        process = await self._start_meter()
        data_pipe = process.stdout
        rows_processed = 0
        # convert gap to microseconds
        self.max_gap = seconds_to_timestamp(max_gap)
        # set up timestamps
        self.clock_ts = time_now()
        self.data_ts = self.clock_ts
        try:
            while(not self.stop_requested):
                if(not disable_alignment):
                    self._align_clock()
                data = await data_pipe.readexactly(BLOCK_SIZE)
                np_data = self._parse(data)
                await output.write(np_data)
                rows_processed += len(np_data)
                if(nrows > 0 and rows_processed >= nrows):
                    self.stop_requested = True
        except asyncio.streams.IncompleteReadError:
            if(not self.stop_requested):
                logging.error("short read from ethstream, exiting")
        finally:
            logging.info("stopping ethstream")
            try:
                process.terminate()
            except ProcessLookupError:
                pass #already terminated ethstream proc

    def stop(self):
        sys.stderr.flush()
        self.stop_requested = True

    async def _start_meter(self):
        cmd = ["ethstream",
               "-a", self.ip_address,
               "-r", "8000",
               "-C", "0,1,2,3,4,5",
               "-L", "-B"]
        create = asyncio.create_subprocess_exec(
            *cmd,
            stdin=asyncio.subprocess.DEVNULL,
            stdout=asyncio.subprocess.PIPE)

        return await create

    def _parse(self, raw):
        if(len(raw) != BLOCK_SIZE):
            raise CaptureError("short read from ethstream process")
        # parse from string: 2 byte units in big endian
        data = np.fromstring(raw, dtype=np.uint16)
        rows = len(raw) // ROW_BYTES
        # give the array the proper shape
        data.shape = (rows, 6)
        # calculate timestamps
        top_ts = self.data_ts + rows * self.data_ts_inc
        ts = np.array(np.linspace(self.data_ts, top_ts,
                                  rows, endpoint=False), dtype=np.uint64)
        ts.shape = (rows, 1)
        self.data_ts = top_ts  # update the timestamp for the next run

        # add timestamps
        ts_data = np.hstack((ts, data))
        # now ts_data is the form
        # [ts, d0, d1, d2, d3, ...]
        # [ts, d0, d1, d2, d3, ...]
        #  ...
        return ts_data

    def _align_clock(self):
        self.clock_ts = time_now()
        if (self.data_ts - self.max_gap) > self.clock_ts:
            print("Data is coming in too fast: data time "
                  "is %s but clock time is only %s" %
                  (timestamp_to_human(self.data_ts),
                   timestamp_to_human(self.clock_ts)))
            exit(1)  # can't recover

        if (self.data_ts + self.max_gap) < self.clock_ts:
            print("Skipping data timestamp forward from "
                  "%s to %s to match clock time\n" % (
                      timestamp_to_human(self.data_ts),
                      timestamp_to_human(self.clock_ts)))
            self.data_ts = self.clock_ts

    def close(self):
        pass  # nothing to do
