import asyncio
import argparse

import numpy as np
import json

from joule import LocalNumpyPipe
from nilm.readers import capture
from nilm.filters import prep, sinefit, reconstructor

SAMPLE_FREQ = 3000


def run(meter, duration):
    """ Given an array of magnetic sensors [mchannels]
        and a single electric sensor [echannel],
        read [duration] seconds of data
        and return prep using identity reconstruction matrix"""
    # figure out how many rows of data to capture
    nrows = duration * 3000  # noncontact sample frequency

    num_sensors = len(meter['sensors']['current']['sensor_indices'])

    # pipes to connect modules
    npipe_raw = LocalNumpyPipe("raw", layout="int16_8")
    npipe_iv = LocalNumpyPipe("iv", layout="float32_%d" % (1 + num_sensors))
    npipe_iv2 = LocalNumpyPipe("iv", layout=npipe_iv.layout)
    npipe_iv.subscribe(npipe_iv2)
    #   prep is P1,Q1 for every magnetic sensor
    prep_layout = "float32_%d" % (num_sensors * 2)
    npipe_prep = LocalNumpyPipe("prep", layout=prep_layout)
    npipe_zc = LocalNumpyPipe("prep", layout="float32_3")

    """
    #pipes for debugging
    tpipe_raw = LocalNumpyPipe("t_raw", npipe_raw.layout, buffer_size=15000)
    npipe_raw.subscribe(tpipe_raw)
    tpipe_iv = LocalNumpyPipe("t_iv", npipe_iv.layout, buffer_size=15000)
    npipe_iv.subscribe(tpipe_iv)
    tpipe_zc = LocalNumpyPipe("t_zc", npipe_zc.layout, buffer_size=15000)
    npipe_zc.subscribe(tpipe_zc)
    """

    # capture reader
    capture_args = argparse.Namespace(**{
        "meter_type": "noncontact",
        "align": False,
        "max_gap": 100,  # ignored
        "nrows": 0,  # run forever
        "tty": "/dev/nilm/%s-data" % meter["serial_number"]
    })

    my_capture = capture.Capture()
    capture_future = asyncio.ensure_future(
        my_capture.run(capture_args, npipe_raw))

    # reconstructor (needed to integrate voltage signal)
    reconstructor_configs = {
        'm_indices': meter['sensors']['current']['sensor_indices'],
        'e_indices': meter['sensors']['voltage']['sensor_index'],
        'max_gap': 1,
        'current_matrix': np.eye(num_sensors).tolist(),
        'integrate': meter['sensors']['voltage']['digitally_integrate'],
        'voltage_matrix': [1.0],
    }
    my_reconstructor = reconstructor.Reconstructor()
    asyncio.ensure_future(
        my_reconstructor.run(
            argparse.Namespace(**reconstructor_configs),
            {'raw': npipe_raw},
            {'iv': npipe_iv}))

    # sinefit
    sinefit_configs = {
        'v_index': 1,
        'frequency': 60,
        'min_freq': 55,
        'max_freq': 65,
        'min_amp': 10
    }
    my_sinefit = sinefit.Sinefit()
    asyncio.ensure_future(
        my_sinefit.run(
            argparse.Namespace(**sinefit_configs),
            {'iv': npipe_iv},
            {'zero_crossings': npipe_zc}))

    # prep
    prep_configs = {
        'nshift': 1,
        'nharm': 1,
        'current_indices': list(range(2, num_sensors + 2)),
        'rotations': np.zeros(num_sensors).tolist(),
        'scale_factor': 1.0,
        'merge': True,
        'polar': False
    }

    my_prep = prep.Prep()
    prep_future = asyncio.ensure_future(
        my_prep.run(
            argparse.Namespace(**prep_configs),
            {'iv': npipe_iv2,
             'zero_crossings': npipe_zc},
            {'prep': npipe_prep}, nrows=nrows))

    loop = asyncio.get_event_loop()

    prep_future.add_done_callback(lambda _: my_capture.stop())
    capture_future.add_done_callback(lambda _: loop.stop())
    loop.run_forever()
    prep_data = npipe_prep.read_nowait()

    """
    raw_data = tpipe_raw.read_nowait()
    iv_data = tpipe_iv.read_nowait()
    zc_data = tpipe_zc.read_nowait()

    ts0 = raw_data['timestamp'][0]
    ts = (raw_data['timestamp']-ts0)/1e6
    plt.plot(ts,raw_data['data'])
    plt.legend(['0','1','2','3','4','5','6','7'])
    plt.title('raw data')
    plt.figure()
    ts = (iv_data['timestamp']-ts0)/1e6
    plt.plot(ts,iv_data['data'])
    plt.legend(['V','1','3','5','7'])
    plt.title('IV data')

    plt.figure()
    plt.plot(ts,iv_data['data'][:,0],label='v')
    ts = (zc_data['timestamp']-ts0)/1e6
    plt.plot(ts,zc_data['data'][:,0],'.',label='zc')
    plt.legend()
    plt.title('sinefit')

    
    plt.figure()

    ts0 = zc_data['timestamp'][0]
    ts_zc = (zc_data['timestamp']-ts0)/1e6
    plt.plot(ts_zc,zc_data['data'][:,0],'.')

    ts_prep = (prep_data['timestamp']-ts0)/1e6
    plt.plot(ts_prep,prep_data['data'])
    plt.legend(['1p','1q','3p','3q','5p','5q','7p','7q'])
    plt.title('prep')
    plt.show()
    """
    return prep_data['data']
