from nilm.cmds.configure import ConfigureCmd
from nilm.meter.utils import Status
import unittest
import tempfile
import argparse
import os
import configparser
from unittest import mock

SINGLE_PHASE_DIR = os.path.join(os.path.dirname(
    __file__),'files', 'single_phase')
DISABLED_METERS_DIR = os.path.join(os.path.dirname(
    __file__),'files', 'disabled_meters')
SEPARATE_PREPS_DIR = os.path.join(os.path.dirname(
    __file__),'files', 'separate_preps')

devnull = open(os.devnull, 'w')


class TestConfigure(unittest.TestCase):

    def test_requests_arguments(self):
        my_cmd = ConfigureCmd(mock.Mock(), mock.Mock())
        p = my_cmd.get_parser("name")
        args = p.parse_args([])
        self.assertEqual(args.stream_dir, "/etc/joule/stream_configs")
        self.assertEqual(args.module_dir, "/etc/joule/module_configs")
        self.assertEqual(args.nilmdb_url, "http://localhost/nilmdb")

    @mock.patch("nilm.cmds.configure.os.path.isdir")
    @mock.patch("nilm.cmds.configure.meter.check_disk_usage")
    @mock.patch("nilm.cmds.configure.Client", autospec=True)
    def test_creates_info_stream_if_required(self, mock_client,
                                             mock_disk_usage, _):
        mock_disk_usage.return_value = Status.ok
        my_cmd = ConfigureCmd(mock.Mock(), mock.Mock())
        my_cmd._build_module_configs = mock.Mock()  # tested below
        my_cmd._build_stream_configs = mock.Mock()  # tested below
        config_file = os.path.join(SINGLE_PHASE_DIR, "meters.yml")
        args = argparse.Namespace(config_file=config_file,
                                  stream_dir="", module_dir="",
                                  nilmdb_url="", legacy=False)
        # does not recreate the info stream if it already exists
        my_client = mock_client.return_value
        my_client.stream_info = mock.Mock(return_value=mock.Mock())
        with mock.patch('sys.stdout', devnull):
            my_cmd.take_action(args)
        my_client.stream_create.assert_not_called()

        # creates the info stream if it does not exist
        my_client.stream_info = mock.Mock(return_value=None)
        with mock.patch('sys.stdout', devnull):
            my_cmd.take_action(args)
        my_client.stream_create.assert_called_with("/meter1/info", "uint8_1")

    @mock.patch("nilm.cmds.configure.os.path.isdir")
    @mock.patch("nilm.cmds.configure.meter.check_disk_usage")
    def test_removes_disabled_meter_configs(self, mock_disk_usage, _):
        mock_disk_usage.return_value = Status.ok
        module_dir = tempfile.TemporaryDirectory()
        # create meter files for disabled meter2
        open(os.path.join(module_dir.name, "meter2_capture.conf"), 'w').close()
        open(os.path.join(module_dir.name, "meter2_process.conf"), 'w').close()
        my_cmd = ConfigureCmd(mock.Mock(), mock.Mock())
        my_cmd._build_info_stream = mock.Mock()  # tested above
        my_cmd._build_module_configs = mock.Mock()  # tested below
        my_cmd._build_stream_configs = mock.Mock()  # tested below
        config_file = os.path.join(DISABLED_METERS_DIR, "meters.yml")
        args = argparse.Namespace(
            config_file=config_file,
            stream_dir="",
            module_dir=module_dir.name,
            legacy=False,
            nilmdb_url="")

        try:
            with mock.patch('sys.stdout', devnull):
                my_cmd.take_action(args)
            self.assertFalse(os.path.exists(
                os.path.join(module_dir.name, "meter2_capture.conf")))
            self.assertFalse(os.path.exists(
                os.path.join(module_dir.name, "meter2_process.conf")))

        except Exception as e:
            raise e
        finally:
            module_dir.cleanup()

    @mock.patch("nilm.cmds.configure.meter.check_disk_usage")
    def test_builds_single_phase_files(self, mock_disk_usage):
        mock_disk_usage.return_value = Status.ok
        my_cmd = ConfigureCmd(mock.Mock(), mock.Mock())
        my_cmd._build_info_stream = mock.Mock()  # tested above
        stream_dir = tempfile.TemporaryDirectory()
        module_dir = tempfile.TemporaryDirectory()
        config_file = os.path.join(SINGLE_PHASE_DIR, "meters.yml")

        args = argparse.Namespace(
            config_file=config_file,
            stream_dir=stream_dir.name,
            module_dir=module_dir.name,
            legacy=False,
            nilmdb_url="")

        try:
            with mock.patch('sys.stdout', devnull):
                my_cmd.take_action(args)
            # --verify modules--
            actual_file = os.path.join(module_dir.name, "%s.conf")
            expected_file = os.path.join(SINGLE_PHASE_DIR,
                                         "module_configs", "%s.conf")
            # verify meter capture
            self._verify_config(expected_file % "meter_capture",
                                actual_file % "meter1_capture")
            # verify meter processing
            self._verify_config(expected_file % "meter_process",
                                actual_file % "meter1_process")
            # --verify streams--
            actual_file = os.path.join(stream_dir.name, "%s.conf")
            expected_file = os.path.join(SINGLE_PHASE_DIR,
                                         "stream_configs", "%s.conf")
            # verify sensor stream
            self._verify_config(expected_file % "sensor",
                                actual_file % "meter1_sensor")
            # verify iv stream
            self._verify_config(expected_file % "iv",
                                actual_file % "meter1_iv")
            # verify sinefit stream
            self._verify_config(expected_file % "sinefit",
                                actual_file % "meter1_sinefit")
            # verify prep stream
            self._verify_config(expected_file % "prep",
                                actual_file % "meter1_prep")

        except Exception as e:
            raise e
        finally:
            stream_dir.cleanup()
            module_dir.cleanup()

    @mock.patch("nilm.cmds.configure.meter.check_disk_usage")
    def test_builds_separated_prep_files(self, mock_disk_usage):
        mock_disk_usage.return_value = Status.ok
        my_cmd = ConfigureCmd(mock.Mock(), mock.Mock())
        my_cmd._build_info_stream = mock.Mock()  # tested above
        stream_dir = tempfile.TemporaryDirectory()
        module_dir = tempfile.TemporaryDirectory()
        config_file = os.path.join(SEPARATE_PREPS_DIR, "meters.yml")

        args = argparse.Namespace(
            config_file=config_file,
            stream_dir=stream_dir.name,
            module_dir=module_dir.name,
            legacy=True,
            nilmdb_url="")

        try:
            with mock.patch('sys.stdout', devnull):
                my_cmd.take_action(args)
            # --verify modules--
            actual_file = os.path.join(module_dir.name, "%s.conf")
            expected_file = os.path.join(SEPARATE_PREPS_DIR,
                                         "module_configs", "%s.conf")
            # verify meter capture
            self._verify_config(expected_file % "meter_capture",
                                actual_file % "meter1_capture")
            # verify meter processing
            self._verify_config(expected_file % "meter_process",
                                actual_file % "meter1_process")
            # --verify streams--
            actual_file = os.path.join(stream_dir.name, "%s.conf")
            expected_file = os.path.join(SEPARATE_PREPS_DIR,
                                         "stream_configs", "%s.conf")
            # verify sensor stream
            self._verify_config(expected_file % "sensor",
                                actual_file % "meter1_sensor")
            # verify iv stream
            self._verify_config(expected_file % "iv",
                                actual_file % "meter1_iv")
            # verify sinefit stream
            self._verify_config(expected_file % "sinefit",
                                actual_file % "meter1_sinefit")
            # verify prep stream
            self._verify_config(expected_file % "prep-a",
                                actual_file % "meter1_prep_a")
            # verify prep stream
            self._verify_config(expected_file % "prep-b",
                                actual_file % "meter1_prep_b")
            # verify prep stream
            self._verify_config(expected_file % "prep-c",
                                actual_file % "meter1_prep_c")

        except Exception as e:
            raise e
        finally:
            stream_dir.cleanup()
            module_dir.cleanup()
            
    def _verify_config(self, expected_file, actual_file):
        expected_config = configparser.ConfigParser()
        expected_config.read(expected_file)
        actual_config = configparser.ConfigParser()
        actual_config.read(actual_file)
        # make sure both configs have the same sections

        self.assertEqual(len(expected_config.sections()),
                         len(actual_config.sections()))
        # check the contents of the sections
        for section in expected_config.sections():
            x = dict(actual_config[section])
            y = dict(expected_config[section])
            self.assertEqual(x, y)
