from joule import LocalPipe
import unittest
import asyncio
import argparse
import numpy as np
from nilm.filters import sinefit


class TestSinefit(unittest.TestCase):

    def test_detects_zero_crossings(self):
        test_voltage = self._create_voltage()
        npipe_iv = LocalPipe(layout="float32_1", name="iv")
        npipe_zc = LocalPipe(layout="float32_3", name="zc")
        npipe_iv.write_nowait(test_voltage)
        args = argparse.Namespace(**{
            "v_index": 1,
            "frequency": 60,
            "min_freq": 40,
            "max_freq": 70,
            "min_amp": 1
        })
        my_filter = sinefit.Sinefit()
        loop = asyncio.new_event_loop()
        asyncio.set_event_loop(loop)
        loop.run_until_complete(my_filter.run(args,
                                              {'iv': npipe_iv},
                                              {'zero_crossings': npipe_zc},
                                              nrows=len(test_voltage) - 200))
        result = npipe_zc.read_nowait()
        expected_zc = 0
        for zc in result:
            np.testing.assert_array_almost_equal(
                np.array([50.0, 1.0, 0.0]), zc['data'])
            self.assertAlmostEqual(zc['timestamp'] / 1e6, expected_zc, delta=2)
            expected_zc += 1 / 50

    def _create_voltage(self, freq=50.0, amp=5.0, t=2, fs=2000):

        t = np.arange(0, t, step=1 / fs)
        v = np.sin(2 * np.pi * t * freq)
        return np.c_[t * 1e6, v]
