#! /usr/bin/python

"""
Command line interface for non-contact sensor calibration

John Donnal 2016
"""

import sys
import subprocess
import os
import time
import serial
import numpy as np
from cliff.command import Command

from nilm.meter import meter
from nilm.meter.utils import MeterConfigurationError
from . import calibrate_phase
from . import calibrate_finalize

METERS_FILE = "/opt/configs/meters.yml"
METER_CAL = "/opt/configs/meters"
CALIB_DATA_DIR = "/opt/configs/cal_data"
TYPE_NON_CONTACT = "noncontact"

ESC_SEQ = "\x1b["
COL_RESET = ESC_SEQ+"0m"
COL_RED = ESC_SEQ+"6;30;41m"
COL_GREEN = ESC_SEQ+"6;30;42m"
COL_YELLOW = ESC_SEQ+"6;30;43m"
COL_BLUE = ESC_SEQ+"34;01m"
COL_MAGENTA = ESC_SEQ+"35;01m"
COL_CYAN = ESC_SEQ+"36;01m"

FNULL = open(os.devnull, 'w')


class CalibrateCmd(Command):
    "calibrate non-contact meters using smart plug"

    def get_parser(self, prog_name):
        parser = super(CalibrateCmd, self).get_parser(prog_name)
        # meter
        parser.add_argument('meter', nargs='?',
                            help="meter from meters.yml config file")
        # --config-file = /opt/configs/meters.yml
        parser.add_argument("--config-file", "-f",
                            default="/opt/configs/meters.yml",
                            dest="config_file",
                            help='file')
        # --visualize = False
        parser.add_argument('--visualize', '-V',
                            help="show plots as calibration runs",
                            action="store_true", default=False)
        # --replay = False
        parser.add_argument('--replay', '-r',
                            help="replay previous calibration",
                            action="store_true", default=False)
        return parser
        
    def take_action(self, parsed_args):

        try:
            meters = meter.load_meters(parsed_args.config_file)

        except MeterConfigurationError as e:
            print("["+COL_RED+"ERROR"+COL_RESET+"]: %s" % e)
            exit(1)

        ########
        # If a meter wasn't specified, list the available ones
        #
        if(parsed_args.meter==None):
            #print out a list of non-contact meters
            self.print_meters(meters)
            exit(0)


        try:
            meter_config = meters[parsed_args.meter]
            if(meter_config["type"]!=TYPE_NON_CONTACT):
                self.print_error("%s is not a non-contact meter" % meter)
                self.print_meters(settings)
                exit(1)
        except KeyError as e:
            print("meter [%s] not found in meters.yml" % e)
            exit(1)

        ########
        #Check directory permissions, if its owned by root, use sudo
        cal_file = METER_CAL+"/"+meter_config["name"]+".yml"
        if (not os.access(METER_CAL,os.W_OK)):
            self.print_error("insufficient permissions on /opt/configs/meters.")
            print("run as root with sudo nilm calibrate ")
            exit(1)
        if os.path.exists(cal_file):
        # Make sure we can write to the meterX file (or use sudo)
            if(not os.access(cal_file,os.W_OK)):
                self.print_error("insufficient permissions on /opt/configs/meters.")
                print("run as root with sudo nilm calibrate ")
                exit(1)
        # Create calibration data directory if necessary
        if not os.path.exists(CALIB_DATA_DIR):
            os.mkdir(CALIB_DATA_DIR)
            
        calibration_config = meter_config["calibration"]
        voltage_configs = meter_config["sensors"]["voltage"]
        current_configs = meter_config["sensors"]["current"]
        phases = meter_config["phases"]
        ########
        # Display config values for user confirmation
        print("")
        if(parsed_args.replay):
            print("**REPLAYING CALIBRATION: run without --replay to actually calibrate**")
            print("")

        print("Calibrating %s"%meter_config["name"])
        # print phase information
        sys.stdout.write( "  + The power system has %d phases and "%phases)
        if(calibration_config["has_neutral"]):
            print("neutral")
        else:
            print("NO neutral")
        # print voltage sensor information
        if(voltage_configs["digitally_integrate"]==False):
            print("  + Using the raw output of sensor %d for voltage measurement"%\
            voltage_configs["sensor_index"])
        else:
            print("  + Digitally integrating sensor %d for voltage measurement"%voltage_configs["sensor_index"])
        # print current sensor information
        print("  + Using sensors %s for current measurements"%\
            ",".join("%d"%x for x in current_configs["sensor_indices"]))
        # print calibration information
        print("  + Calibration load is %dW and will run for %d seconds"%\
            (calibration_config["watts"],calibration_config["duration"]))
        print("  + The reference voltage is %dV rms"%\
            voltage_configs["nominal_rms_voltage"])
        print("  + The meter serial number is [%s]"%meter_config["serial_number"])
        # print visualization setting
        if(parsed_args.visualize):
            print("  + Visualization is [ON]")
        else:
            print("  + Visualization is [OFF]")

        # make sure this information makes sense
        if(not calibration_config['has_neutral'] and meter_config['phases']!=3):
            print("\n[ERROR] only a 3 phase system may not have a neutral")
            exit(1)
        if(meter_config['phases']>len(current_configs["sensor_indices"])):
            print("\n[ERROR] current sensors must be >= phases")
            exit(1)
        # ask for confirmation"
        response = input("Is this correct? (y/n) ")

        if(response!='y'):
            print("Calibration cancelled, exiting")
            exit(0)
        # check if plug should be setup for calibration
        valid = False
        while(not valid):
            response = input("Set up a smart plug for calibration? (y/n) ")
            if(response != 'y' and response != 'n'):
                print("Please answer y or n")
            else:
                valid = True
        ##########
        # Stop the nilm-capture service if it is running
        ret = subprocess.call(["pgrep","jouled"], stdout=FNULL, stderr=FNULL)
        if(ret==0):
            subprocess.call(["sudo","service","jouled","stop"],
                            stdout=FNULL, stderr=FNULL)
            print("waiting 15 seconds for joule daemon to stop")
            time.sleep(15)
            restart_capture = True
        else:
            restart_capture = False

        # ...it should, loop until user cancels or plug is set up
        if(response=='y'):
            self.configure_plug()
        #Reset the calibration config values
        num_phases = meter_config["phases"]
        num_sensors = len(current_configs["sensor_indices"])
        if(not calibration_config["has_neutral"]):
            num_phases -= 1
        calibration_config["sinefit_rotations"] = np.zeros(num_phases)
        calibration_config["sensor_matrix"] = np.zeros((num_sensors, num_phases))
        calibration_config["current_matrix"] = np.zeros((num_phases, num_sensors))
        #pq_coeffs is the norm_coeffs matrix its NxSx2 where
        #N=phases and S=sensors
        calibration_config["pq_coeffs"] = np.zeros((num_phases,num_sensors,2))

        #Now calibrate each phase
        num_phases = meter_config["phases"]
        if(calibration_config["has_neutral"]==False):
            num_phases -= 1 #only need 2 line-line runs
        print("Connect the calibration load to the first phase")
        for i in range(num_phases):
            response = input("Once it is running press [ENTER] to continue ")
            while True:
                (exit_code,msg) = calibrate_phase.run(i,meter_config,replay=parsed_args.replay,debug=parsed_args.visualize)
                if(exit_code!=0):
                    response = input("  Try again? (y/n) ")
                    if(response!='y'):
                        print("Calibration cancelled, exiting")
                        exit(1)
                else:
                    break
            if(i<(num_phases-1)):
                print("Move the calibration load to the next phase")
        #Finalize the calibration
        response = input("Phase calibration complete, save results? (y/n) ")
        if(response=='y'):
            calibrate_finalize.run(meter_config,debug=parsed_args.visualize)
        else:
            print("Calibration cancelled")
        if(restart_capture):
            print("restarting nilm data capture")
            subprocess.call(["sudo","service","jouled","start"],
                            stdout=FNULL, stderr=FNULL)
        else:
            print("to start data capture run [sudo service jouled start]")
        print("***ALL DONE***")

    def print_meters(self, settings):
        if(settings is None):
            return
        print("List of non-contact meters:")
        has_noncontact=False
        for meter in settings:
            if(settings[meter]["type"]==TYPE_NON_CONTACT):
                print("\t+ %s: serial number [%s]"%(meter,settings[meter]["serial_number"]))
                has_noncontact = True
        if(not has_noncontact):
            print("\t no available noncontact meters to calibrate")
        print("usage: sudo nilm calibrate meterX")

    def configure_plug(self):
        try:
            from nilmplug.plug import usb_plug
        except ImportError:
            self.print_error("install the nilmplug module")
            exit(1)
            
        success=False
        while(success==False):
            sys.stdout.write("  + configuring plug\n")
            try:
                usb_plug.set_calibrate(True,1000,2000)
                success=True
            except serial.SerialException as e:
                self.print_error("plug not available")
                print("  Make sure plug is connected and the light is green")
                response = input("  Try again? (y/n) ")
                if(response!='y'):
                    print("Calibration cancelled, exiting")
                    exit(1)
        print("  [OK]")

    
    def print_error(self, msg):
        print("["+COL_RED+"ERROR"+COL_RESET+"] %s" % msg)


    

