import unittest
from unittest import mock
from nilm.meter import meter
from nilm.meter.utils import MeterConfigurationError
import os
import sys
from contextlib import contextmanager
from io import StringIO

CAL_DIR = os.path.join(os.path.dirname(
    __file__), 'files', 'default_cal')

devnull = open(os.devnull, 'w')


class TestMeterErrors(unittest.TestCase):

    @mock.patch("nilm.meter.storage.requests.get", autospec=True)
    def setUp(self, client):
        self.instance = client.return_value
        # values from bucket
        self.instance.dbinfo = mock.Mock(
            return_value={"path":     '/home/nilmdb',
                          "other":    9180803731456,
                          "reserved": 147308150784,
                          "free":     978309853184,
                          "size":     4364062072832})

    def test_handles_bad_keep_setting(self):
        trials = ["invalid_keep", "bad_keep", "missing_keep"]
        for trial in trials:
            meter_file = self.gen_meter_config(trial)
            with self.captured_output() as (out, err):
                with self.assertRaises(MeterConfigurationError):
                    meter.load_meters(meter_file, CAL_DIR)

    def test_handles_bad_decimate_setting(self):
        trials = ["invalid_decimate", "missing_decimate"]
        for trial in trials:
            meter_file = self.gen_meter_config(trial)
            with self.captured_output() as (out, err):
                with self.assertRaises(MeterConfigurationError):
                    meter.load_meters(meter_file, CAL_DIR)
                
    def gen_meter_config(self, name):
        return os.path.join(os.path.dirname(
            __file__), 'files', 'bad_configs', name+'.yml')

    @contextmanager
    def captured_output(self):
        new_out, new_err = StringIO(), StringIO()
        old_out, old_err = sys.stdout, sys.stderr
        try:
            sys.stdout, sys.stderr = new_out, new_err
            yield sys.stdout, sys.stderr
        finally:
            sys.stdout, sys.stderr = old_out, old_err

