from nilm.meter.utils import print_error, Status
from nilm.meter.streams import check_stream_config

import re

# Check syntax and settings for a non-contact meter
#


def check_syntax(name, config):
    try:
        # 1 check the enabled flag
        x = config["enabled"]
        if((not type(x) is bool)):
            print_error(
                "%s invalid [enabled] setting, must be [true|false]" % name)
            return Status.error

        # 2 check phases
        x = config["phases"]
        if(not (type(x) is int) or (
                x != 1 and x != 2 and x != 3)):
            print_error("%s invalid [phase] setting, must be [1|2|3]" % (name),
                        "software#software_configuration")
            return Status.error
        phases = x  # to compare with number of current sensors
        # 3 check serial_number
        x = config["serial_number"]
        if((not (type(x) is str)) or re.match("^meter\w+$", x) == None):
            print_error("%s invalid [serial_number] must be [meterXXXX]" % name,
                        "software#software_configuration")
            return Status.error
        # 4 check calibration
        cal_config = config["calibration"]
        if(not(type(cal_config) is dict)):
            print_error("%s invalid [calibration] settings" % name)
            return Status.error
        try:
            # 4a check duration
            x = cal_config["duration"]
            if((not type(x) is int) or x < 3):
                print_error(
                    "%s invalid calibration [duration], must be > 3" % name)
                return Status.error
            # 4b check watts
            x = cal_config["watts"]
            if((not type(x) is int) or x <= 0):
                print_error(
                    "%s invalid calibration [watts], must be > 0" % name)
                return Status.error
            # 4c check has_neutral flag
            x = cal_config["has_neutral"]
            if((not type(x) is bool)):
                print_error(
                    "%s invalid calibration [has_neutral] must be [true|false]" % name)
                return Status.error
        except KeyError as e:
            print_error(
                "%s error in [calibration]: missing [%s]" % (name, e))
            return Status.error
        # 5 check sensors
        sensor_config = config["sensors"]
        if(not(type(sensor_config) is dict)):
            print_error("%s invalid [sensors] settings" % name)
            return Status.error
        try:
            # 5a check voltage
            voltage_config = sensor_config["voltage"]
            if(not(type(voltage_config) is dict)):
                print_error("%s invalid [sensors][voltage] settings" % name)
                return Status.error
            try:
                # 5a1 check sensor_index
                x = voltage_config["sensor_index"]
                if((not type(x) is int) or (x < 0) or (x > 7)):
                    print_error(
                        "%s invalid [sensors][voltage][sensor_index] must be [0-7]" % name)
                    return Status.error
                voltage_index = x  # save to compare with current indices
                # 5a2 check digitally_integrate
                x = voltage_config["digitally_integrate"]
                if((not type(x) is bool)):
                    print_error("%s invalid [sensors][voltage][digitally_integrate] must be [true|false]" %
                                name)
                    return Status.error
                # 5a3 check nominal_rms_voltage
                x = voltage_config["nominal_rms_voltage"]
                if((not type(x) is int) and (not type(x) is float)):
                    print_error(
                        "%s invalid [sensors][voltage][nominal_rms_voltage] must be a number" % name)
                    return Status.error
                if(x <= 0):
                    print_error(
                        "%s invalid [sensors][voltage][nominal_rms_voltage] must be > 0" % name)
                    return Status.error
            except KeyError as e:
                print_error(
                    "%s error in [sensors][voltage]: missing [%s]" % (name, e))
                return Status.error
            # 5b check current
            current_config = sensor_config["current"]
            if(not(type(current_config) is dict)):
                print_error("%s invalid [sensors][current] settings" % name)
                return Status.error
            try:
                # 5b1 check sensor_indices
                x = current_config["sensor_indices"]
                if((not type(x) is list)):
                    print_error(
                        "%s invalid [sensors][current][sensor_indices] must be a list" % name)
                    return Status.error
                for val in x:
                    # 5b1 check sensor_indices are valid ints
                    if(not(type(val) is int) or (val < 0) or (val > 7)):
                        print_error("%s invalid value in [sensors][current][sensor_indices] \n\t" % name +
                                    "must be between 0 and 7")
                        return Status.error
                    # 5b1 make sure the voltage sensor is different than the
                    # current sensors
                    if(val == voltage_index):
                        print_error("%s sensor [%d] cannot be both a voltage and a current sensor" %
                                    (name, val))
                        return Status.error
                # 5b1 make sure values in sensor_indices are unique
                if(len(set(x)) != len(x)):
                    print_error("%s invalid values in [sensors][current][sensor_indices] \n\t" % name +
                                "repeats are not allowed")
                    return Status.error
                # 5b1 make sure there are enough current sensors
                effective_phases = phases
                if(config["calibration"]["has_neutral"] is False):
                    effective_phases -= 1
                if(effective_phases > len(x)):
                    print_error(
                        "%s number of current sensors must be greater than or equal to phases" % name)
                    return Status.error
            except KeyError as e:
                print_error(
                    "%s error in [sensors][current], missing [%s]" % (name, e))
                return Status.error
        except KeyError as e:
            print_error("%s error in [sensors]: missing [%s]" % (name, e))
            return Status.error
        # 6 check streams
        stream_config = config["streams"]
        if(not(type(stream_config) is dict)):
            print_error("%s invalid [streams] settings" % name)
            return Status.error
        try:
            r = min(0, check_stream_config(
                name, "sinefit", stream_config["sinefit"]))
            r = min(r, check_stream_config(name, "iv", stream_config["iv"]))
            r = min(r, check_stream_config(
                name, "prep", stream_config["prep"]))
            r = min(r, check_stream_config(
                name, "sensor", stream_config["sensor"]))
            if(r != Status.ok):
                return Status.error  # check_stream_config prints the error message
        except KeyError as e:
            print_error("%s error in [streams]: missing [%s]" % (name, e))
    except KeyError as e:
        print_error("%s missing configuration [%s]" % (name, e))
        return Status.error
    return Status.ok
