"""
Reads data from usb or ethstream device
"""

from joule import ReaderModule
from joule.utilities import yesno
from .serialcapture import SerialCapture
from .ethstreamcapture import EthstreamCapture
from .errors import CaptureError

ARGS_DESC = """
---
:name:
  NILM Reader
:author:
  John Donnal
:license:
  Closed
:url:
  http://git.wattsworth.net/wattsworth/nilm.git
:usage:
  Capture data from NILM acquisition devices (contact or non-contact)
    
  | Arguments            | Default | Description
  |----------------------|---------|-------------------------------------------------
  |``max-gap``           | 10      | max seconds between sample ts and clock
  |``align-with-clock`` | yes      | [**yes**,**no**] align samples with system clock
  |``meter-type``        | --      | [**contact**,**noncontact**] required parameter
  |``ip-address``        | 192.168.1.209      | contact meter IP address
  |``tty``               | --      | non-contact meter serial port
  |``nrows``             | 0       | read n rows and exit (0=read forever)

:inputs:
  None

:outputs:
  Noncontact Meter:

  output
  : ``int16`` with 8 elements @ 3kHz

  Contact Meter:

  output
  : ``uint16`` with 6 elements @ 8kHz

:stream_configs:
  #output (noncontact)#
     [Main]
     name = Raw
     path = /path/to/output
     datatype = int16
     keep = 1w

     # for D-Board configuration
     [Element1]
     name = dE-Field 1
     [Element2]
     name = B-Field 1
     [Element3]
     name = dE-Field 2
     [Element4]
     name = B-Field 2
     [Element5]
     name = dE-Field 3
     [Element6]
     name = B-Field 3
     [Element7]
     name = dE-Field 4
     [Element8]
     name = B-Field 4

  #output (contact)#
     [Main]
     name = Raw
     path = /path/to/output
     datatype = uint16
     keep = 1w

     # data is in ADC counts
     [Element1]
     name = Current A
     [Element2]
     name = Current B
     [Element3]
     name = Current C
     [Element4]
     name = Voltage A
     [Element5]
     name = Voltage B
     [Element6]
     name = Voltage C

:module_config:
    [Main]
    name = NILM Reader
    exec_cmd = nilm-reader

    [Arguments]
    max-gap = 10
    align-with-clock = yes
    meter-type = noncontact
    tty = /dev/nilm/meter001-data

    [Outputs]
    output = /path/to/output
---
"""


class Capture(ReaderModule):
        
    def custom_args(self, parser):
        # --max-gap = 10
        parser.add_argument("--max-gap",
                            default="10", type=int,
                            help="max misalignment between sample timestamp and system clock (in seconds)")
        # --disable-alignment = False
        parser.add_argument("--align-with-clock",
                            default=True, type=yesno,
                            dest="align",
                            help="align timestamps to system clock")
        # --meter-type
        parser.add_argument("--meter-type",
                            required=True,
                            dest="meter_type",
                            choices=["contact", "noncontact"])
        # --ip-address
        parser.add_argument("--ip-address",
                            default="192.168.1.209")
        # --tty
        parser.add_argument("--tty",
                            help="noncontact tty file (eg /dev/nilm/meterXXXX-data")
        # --nrows
        parser.add_argument("--nrows", type=int, default=0,
                            help="read n rows and exit (0=read forever)")
                            
        parser.description = ARGS_DESC
        
    async def run(self, parsed_args, output):
        self.my_capture = None
        try:

            if (parsed_args.meter_type == 'noncontact'):
                if (parsed_args.tty is None):
                    raise Exception("[tty] parameter required for noncontact meters")

                self.my_capture = SerialCapture(parsed_args.tty)
            else:
                self.my_capture = EthstreamCapture(parsed_args.ip_address)
            await self.my_capture.run(output,
                                      max_gap=parsed_args.max_gap,
                                      align=parsed_args.align,
                                      nrows=parsed_args.nrows)
        except CaptureError as e:
            print(str(e))
        finally:
            if self.my_capture is not None:
                self.my_capture.close()
        await output.close()

    def stop(self):
        self.my_capture.stop()

        
def main():
    r = Capture()
    r.start()

    
if __name__ == "__main__":
    main()
