
"""
Configures joule files based off meters.yml
"""

import configparser
import os
import sys
from cliff.command import Command
from nilm.meter import meter
from nilm.meter.utils import MeterConfigurationError

ESC_SEQ = "\x1b["
COL_RESET = ESC_SEQ+"0m"
COL_RED = ESC_SEQ+"6;30;41m"
COL_GREEN = ESC_SEQ+"6;30;42m"
COL_YELLOW = ESC_SEQ+"6;30;43m"
COL_BLUE = ESC_SEQ+"34;01m"
COL_MAGENTA = ESC_SEQ+"35;01m"
COL_CYAN = ESC_SEQ+"36;01m"


class ConfigureCmd(Command):
    " create joule modules from meters.yml "
    
    def __init__(self, app, app_args, cmd_name=None):
        super(ConfigureCmd, self).__init__(app, app_args, cmd_name)

    def get_parser(self, prog_name):
        parser = super(ConfigureCmd, self).get_parser(prog_name)
        # --config-file = /opt/configs/meters.yml
        parser.add_argument("--config-file", "-f",
                            default="/opt/configs/meters.yml",
                            dest="config_file",
                            help='file')
        # --stream-configs = /etc/joule/stream_configs
        parser.add_argument("--stream-configs", "-s",
                            default="/etc/joule/stream_configs",
                            dest="stream_dir")
        # --module-configs = /etc/joule/module_configs
        parser.add_argument("--module-configs", "-m",
                            default="/etc/joule/module_configs",
                            dest="module_dir")
        # --nimdb-url = http://localhost/nilmdb
        parser.add_argument("--nilmdb-url", "-n",
                            default="http://localhost/nilmdb",
                            dest="nilmdb_url")
        # --legacy = False (PQ with separate preps)
        parser.add_argument("--legacy",
                            action="store_true",
                            dest="legacy")
        return parser

    def take_action(self, parsed_args):
        sys.stdout.write("Reading meter configuration ")
        try:
            meters = meter.load_meters(parsed_args.config_file)
        except MeterConfigurationError as e:
            print("["+COL_RED+"ERROR"+COL_RESET+"]\n invalid meter configuration")
            exit(1)
        # make sure stream_configs and module_configs exist
        if(not os.path.isdir(parsed_args.module_dir)):
            print("["+COL_RED+"ERROR"+COL_RESET+"]\n invalid module_dir, specify with -m")
            exit(1)
        if(not os.path.isdir(parsed_args.stream_dir)):
            print("["+COL_RED+"ERROR"+COL_RESET+"]\n invalid stream_dir, specify with -s")
            exit(1)
        self.module_dir = parsed_args.module_dir
        self.stream_dir = parsed_args.stream_dir
        print("["+COL_GREEN+"OK"+COL_RESET+"]")
                
        # Build the meters
        for my_meter in meters.values():
            if(my_meter['enabled']):
                print(my_meter['name']+":")
                self._build_module_configs(my_meter, parsed_args.legacy)
                self._build_stream_configs(my_meter, parsed_args.legacy)
            else:
                print(my_meter['name']+": (disabled)")
                self._remove_modules(my_meter)
            
    def _build_module_configs(self, my_meter, legacy):
        sys.stdout.write("\tgenerating module configuration ")

        path = my_meter['path']
        sensor_path = "%s/sensor" % path
        iv_path = "%s/IV" % path
        sf_path = "%s/sinefit" % path
        prep_path = "%s/prep" % path

        # --meter capture configuration--
        capture_config = configparser.ConfigParser()
        capture_config['Main'] = {"name": "%s capture" % my_meter['name'],
                                  "exec_cmd": "nilm-reader-capture"}
        if(my_meter["type"] == 'contact'):
            capture_config['Main']['description'] = 'ethstream capture from LabJack UE9'
        else:
            capture_config['Main']['description'] = 'serial data capture'
        capture_config['Arguments'] = {
            "max-gap": 10,
            "align-with-clock": "yes",
            "meter-type": my_meter["type"],
        }
        if(my_meter["type"] == "contact"):
            capture_config['Arguments']['ip-address'] = my_meter["ip_address"]
            capture_config['Arguments']['daq-type'] = my_meter["daq_type"]
            if(my_meter['daq_type'] == "nerdjack"):
                if(my_meter["sensors"]["filter"]["keep"] == True):
                    capture_config['Arguments']['filter-index'] = "%i"%my_meter["sensors"]["filter"]["filter_index"]
                else:
                    capture_config['Arguments']['filter-index'] = None
            
        else:
            capture_config['Arguments']['tty'] = my_meter["tty"]

        capture_config['Inputs'] = {}
        capture_config['Outputs'] = {"output": sensor_path}
        path = os.path.join(self.module_dir, "%s_capture.conf" % my_meter['name'])
        try:
            with open(path, 'w') as configfile: 
                capture_config.write(configfile)
        except PermissionError:
            self.run_as_root()
        # --meter process configuration--
        process_config = configparser.ConfigParser()
        process_config['Main'] = {"name": "%s process" % my_meter['name'],
                                  "description":
                                  "reconstruct -> sinefit -> prep",
                                  "exec_cmd": "nilm-filter-rawtoprep"}
        process_config['Arguments'] = {
            'meter': my_meter["name"],
            'frequency': 60,
            'min-voltage': 10
        }
        if(legacy):
            # PQ with separate stream for each phase
            process_config['Arguments']['merge'] = "no"
            process_config['Arguments']['polar'] = "no"
        else:
            # PQ with one combined stream
            process_config['Arguments']['merge'] = "yes"
            process_config['Arguments']['polar'] = "no"

        process_config['Inputs'] = {"raw": sensor_path}
        process_config['Outputs'] = {"iv": iv_path,
                                     "zero_crossings": sf_path}

        if legacy:
            for n in range(my_meter['phases']):
                ph = 'abc'[n]
                stream = "%s-%c" % (prep_path, ph)
                process_config['Outputs']['prep-%c' % ph] = stream
        else:
            process_config['Outputs']["prep"] = prep_path
            
        path = os.path.join(self.module_dir, "%s_process.conf" % my_meter['name'])
        with open(path, 'w') as configfile:
            process_config.write(configfile)
        print("["+COL_GREEN+"OK"+COL_RESET+"]")

    def _build_stream_configs(self, my_meter, legacy):
        sys.stdout.write("\tgenerating stream configuration ")
        path = my_meter['path']
        name = my_meter['name']
        # --sensor stream--
        settings = my_meter["streams"]["sensor"]
        if(my_meter["type"] == "noncontact"):
            dtype = "int16"
        else:  # contact meter
            dtype = "uint16"
        sensor_config = configparser.ConfigParser()
        sensor_config['Main'] = {"name": "sensor",
                                 "datatype": dtype,
                                 "keep": settings["keep"],
                                 "decimate": settings["decimate"],
                                 "path": path}
        if(my_meter['type'] == "noncontact"):
            num_channels = 8
        else:  # contact
            if(my_meter['daq_type'] == "labjack"):
                num_channels = 6
            if(my_meter['daq_type'] == "nerdjack"):
                if(my_meter["sensors"]["filter"]["keep"] == True):
                    num_channels = 7
                else:
                    num_channels = 6

        for i in range(1, num_channels + 1):
            sensor_config['Element%d' % i] = {"name": "Ch%i" % i,
                                              "units": "ADC"}
            fpath = os.path.join(self.stream_dir, "%s_sensor.conf" % name)
        with open(fpath, 'w') as configfile:
            sensor_config.write(configfile)

        # --iv stream--
        settings = my_meter["streams"]["iv"]
        iv_config = configparser.ConfigParser()
        iv_config['Main'] = {"name": "IV",
                             "datatype": "float32",
                             "keep": settings["keep"],
                             "decimate": settings["decimate"],
                             "path": path}
        i = 1
        if(my_meter["type"] == "noncontact"):
            iv_config['Element1'] = {"name": "Voltage",
                                     "units": "V"}
            i += 1
        else:  # contact
            for ph in range(my_meter["phases"]):
                ph_name = ['A', 'B', 'C'][ph]
                iv_config['Element%d' % i] = {
                    "name": "%c V" % ph_name,
                    "units": "V"}
                i += 1

        for ph in range(my_meter["phases"]):
            ph_name = ['A', 'B', 'C'][ph]
            iv_config['Element%d' % i] = {"name": "%c I" % ph_name,
                                          "units": "A"}
            i += 1
        fpath = os.path.join(self.stream_dir, "%s_iv.conf" % name)
        with open(fpath, 'w') as configfile:
            iv_config.write(configfile)

        # --sinefit stream--
        settings = my_meter["streams"]["sinefit"]
        sf_config = configparser.ConfigParser()
        sf_config['Main'] = {"name": "sinefit",
                             "datatype": "float32",
                             "keep": settings["keep"],
                             "decimate": settings["decimate"],
                             "path": path}
        sf_config['Element1'] = {"name": "Frequency",
                                 "units": "Hz",
                                 "discrete": "yes"}
        sf_config['Element2'] = {"name": "Amplitude",
                                 "units": "V",
                                 "discrete": "yes"}
        sf_config['Element3'] = {"name": "Offset",
                                 "units": "V",
                                 "discrete": "yes"}
        fpath = os.path.join(self.stream_dir, "%s_sinefit.conf" % name)
        with open(fpath, 'w') as configfile:
            sf_config.write(configfile)

        # --prep stream--
        if(legacy):
            self._separate_prep_streams(my_meter)
        else:
            self._merged_prep_stream(my_meter)
            
        print("["+COL_GREEN+"OK"+COL_RESET+"]")

    def _merged_prep_stream(self, my_meter):
        name = my_meter['name']
        path = my_meter['path']
        settings = my_meter["streams"]["prep"]
        prep_config = configparser.ConfigParser()
        prep_config['Main'] = {"name": "prep",
                               "datatype": "float32",
                               "keep": settings["keep"],
                               "decimate": settings["decimate"],
                               "path": path}
        i = 1
        for ph in range(my_meter["phases"]):
            ph_name = ['A', 'B', 'C'][ph]
            for harm in [1, 3, 5, 7]:
                p_name = "%c P%d" % (ph_name, harm)
                prep_config['Element%d' % i] = {"name": p_name,
                                                "units": "W"}
                i += 1
                q_name = "%c Q%d" % (ph_name, harm)
                prep_config['Element%d' % i] = {"name": q_name,
                                                "units": "W"}
                i += 1
        fpath = os.path.join(self.stream_dir, "%s_prep.conf" % name)
        with open(fpath, 'w') as configfile:
            prep_config.write(configfile)

    def _separate_prep_streams(self, my_meter):
        name = my_meter['name']
        path = my_meter['path']
        settings = my_meter["streams"]["prep"]
        for n in range(my_meter["phases"]):
            ph = 'abc'[n]
            prep_config = configparser.ConfigParser()
            prep_config['Main'] = {"name": "prep-%c" % ph,
                                   "datatype": "float32",
                                   "keep": settings["keep"],
                                   "decimate": settings["decimate"],
                                   "path": path}
            i = 1
            for harm in [1, 3, 5, 7]:
                prep_config['Element%d' % i] = {"name": "P%d" % harm,
                                                "units": "W"}
                prep_config['Element%d' % (i+1)] = {"name": "Q%d" % harm,
                                                    "units": "W"}
                i += 2
            fpath = os.path.join(self.stream_dir,
                                "%s_prep_%c.conf" % (name, ph))
            with open(fpath, 'w') as configfile:
                prep_config.write(configfile)

    def _remove_modules(self, my_meter):
        sys.stdout.write("\tremoving module configurations ")
        capture_file = os.path.join(self.module_dir,
                                    "%s_capture.conf" % my_meter['name'])
        if(os.path.isfile(capture_file)):
            os.remove(capture_file)
        process_file = os.path.join(self.module_dir,
                                    "%s_process.conf" % my_meter['name'])
        if(os.path.isfile(process_file)):
            os.remove(process_file)
        print("["+COL_GREEN+"OK"+COL_RESET+"]")
        
    def run_as_root(self):
        print("["+COL_RED+"ERROR"+COL_RESET+"]\n run as [sudo nilm configure]")
        exit(1)