import socket  # for checking valid IP addr
from nilm.meter.streams import check_stream_config
from nilm.meter.utils import print_error, print_warning, Status
import numpy as np

# Check syntax and settings for a contact meter
#


def check_syntax(name, config):
    my_status = Status.ok
    try:
        # 1 check the enabled flag
        x = config["enabled"]
        if((not type(x) is bool)):
            print_error(
                "%s invalid [enabled] setting, must be [true|false]" % name)
            return Status.error

        # 2 check phases
        x = config["phases"]
        if(not (type(x) is int) or (
                x != 1 and x != 2 and x != 3)):
            print_error("%s invalid [phase] setting, must be [1|2|3]" % (name),
                        "software#software_configuration")
            return Status.error
        phases = x  # to compare with number of current sensors

        # 3 check ip_address
        x = config["ip_address"]
        if(not (type(x) is str)):
            print_error("%s invalid [ip_address]" % name)
            return Status.error
        try:
            socket.inet_aton(x)
        except socket.error:
            print_error("%s invalid [ip_address]" % name)
            return Status.error

        # 4 check sensors
        sensor_config = config["sensors"]
        if(not(type(sensor_config) is dict)):
            print_error("%s invalid [sensors] settings" % name)
            return Status.error
        try:
            # 4a check voltage
            voltage_config = sensor_config["voltage"]
            if(not(type(voltage_config) is dict)):
                print_error("%s invalid [sensors][voltage] settings" % name)
                return Status.error
            try:
                # 4a1 check sensor_indices
                x = voltage_config["sensor_indices"]
                voltage_indices = x  # to compare against the current sensor_indices
                # 4a1 must be a list
                if(not(type(x) is list)):
                    print_error(
                        "%s invalid [sensors][voltage][sensor_indices], must be a list" % name)
                    return Status.error
                # 4a1 must be numbers 0-6 (warn if not 3,4 or 5)
                for val in x:
                    if(not(type(val) is int) or (val < 0) or (val > 5)):
                        print_error("%s invalid value in [sensors][voltage][sensors_indices],\n\t" % name +
                                    "must be between 0 and 5")
                        return Status.error
                    if(val < 3):  # 0,1,2
                        print_warning("%s unexpected value [%d] in [sensors]" % (name, val) +
                                      "[voltage][sensor_indices],\n\t" +
                                      "voltage sensors are 3,4,5 by default")
                        my_status = min(my_status, Status.warn)
                # 4a1 must be unique
                if(len(set(x)) != len(x)):
                    print_error("%s invalid values in [sensors][voltage][sensor_indices] \n\t" % name +
                                "repeats are not allowed")
                    return Status.error
                # 4a1 must match # phases
                if(len(x) != phases):
                    print_error("%s number of sensors in [sensors][voltage][sensor_indices]" % name +
                                " must match [phases]")
                    return Status.error
                # 4a2 check sensor_scales
                x = voltage_config["sensor_scales"]
                # 4a2 must be a number or a list
                if(type(x) is list):
                    # 4a2a must be a list of numbers
                    for val in x:
                        if(not(type(val) is int) and not(type(val) is float)):
                            print_error("%s invalid value in [sensors][voltage][sensor_scales]" % name +
                                        "\n\tmust be numbers")
                            return Status.error
                    # 4a2b length must match number of phases
                    if(len(x) != phases):
                        print_error("%s if a list is specified in [sensors][voltage][sensor_scales]," % name +
                                    "\n\tit must match the number of voltage sensors")
                        return Status.error
                #... if it is a number
                elif((type(x) is int) or (type(x) is float)):
                    # 4a2a warn if the value is not 0.0919
                    if(x != 0.0919):
                        print_warning("%s unexpected value for [sensors][voltage][sensor_scales]," % name +
                                      "\n\t default is 0.0919")
                        my_status = min(my_status, Status.Warn)
                #... it is not a number or a list, invalid
                else:
                    print_error("%s invalid setting for [sensors][voltage][sensor_scales]" % name +
                                "\n\tmust be a number or a list of numbers")
                    return Status.error
                # 4a3 check sinefit_phase
                x = voltage_config["sinefit_phase"]
                valid_phases = "ABC"[:phases]
                phase_string = ",".join(list(valid_phases))
                # 4a3 must be a char
                if(not(type(x) is str) or len(x) != 1):
                    print_error("%s invalid setting for [sensors][voltage][sinefit_phase]" % name +
                                "\n\tmust be [%s]" % phase_string)
                    return Status.error
                # 4a3 must match one of the valid phases
                matched = False
                for y in valid_phases:
                    if x == y:
                        matched = True
                if(matched == False):
                    print_error("%s invalid setting for [sensors][voltage][sinefit_phase]" % name +
                                "\n\tmust be [%s]" % phase_string)
                    return Status.error
                # 4a4 check nominal_rms_voltage
                x = voltage_config["nominal_rms_voltage"]
                if((not type(x) is int) and (not type(x) is float)):
                    print_error(
                        "%s invalid [sensors][voltage][nominal_rms_voltage] must be a number" % name)
                    return Status.error
                if(x <= 0):
                    print_error(
                        "%s invalid [sensors][voltage][nominal_rms_voltage] must be > 0" % name)
                    return Status.error
            except KeyError as e:
                print_error(
                    "%s error in [sensors][voltage]: missing [%s]" % (name, e))
                return Status.error

            # 4b check current
            current_config = sensor_config["current"]
            if(not(type(current_config) is dict)):
                print_error("%s invalid [sensors][current] settings" % name)
                return Status.error
            try:
                # 4b1 check sensor_indices
                x = current_config["sensor_indices"]
                # 4b1 must be a list
                if(not(type(x) is list)):
                    print_error(
                        "%s invalid [sensors][current][sensor_indices], must be a list" % name)
                    return Status.error
                # 4b1 must be numbers 0-6 (warn if not 0, 1, or 2)
                for val in x:
                    if(not(type(val) is int) or (val < 0) or (val > 5)):
                        print_error("%s invalid value in [sensors][current][sensors_indices],\n\t" % name +
                                    "must be between 0 and 5")
                        return Status.error
                    if(val >= 3):  # 3,4,5
                        print_warning("%s unexpected value [%d] in [sensors]" % (name, val) +
                                      "[current][sensor_indices],\n\t" +
                                      "current sensors are 0,1,2 by default")
                        my_status = min(my_status, Status.warn)
                # 4b1 must be unique
                if(len(set(x)) != len(x)):
                    print_error("%s invalid values in [sensors][current][sensor_indices] \n\t" % name +
                                "repeats are not allowed")
                    return Status.error
                # 4b1 must match # phases
                if(len(x) != phases):
                    print_error("%s number of sensors in [sensors][current][sensor_indices]" % name +
                                " must match [phases]")
                    return Status.error
                # 4b1 no overlap between current and voltage sensors
                total_sensors = set(x + voltage_indices)
                if(len(total_sensors) != (len(x) + len(voltage_indices))):
                    print_error("%s each sensor must be a voltage OR a current, at least one has" % name +
                                "\n\tbeen used for both [sensor_indices]")
                    return Status.error
                # 4b2 check sensor_scales
                x = current_config["sensor_scales"]
                # 4b2 must be a number or a list
                if(type(x) is list):
                    # 4b2a must be a list of numbers
                    for val in x:
                        if(not(type(val) is int) and not(type(val) is float)):
                            print_error("%s invalid value in [sensors][current][sensor_scales]" % name +
                                        "\n\tmust be numbers")
                            return Status.error
                    # 4b2b length must match number of phases
                    if(len(x) != phases):
                        print_error("%s if a list is specified in [sensors][current][sensor_scales]," % name +
                                    "\n\tit must match the number of current sensors")
                        return Status.error
                #... if it is a number
                elif((type(x) is int) or (type(x) is float)):
                    pass  # Status.ok
                #... it is not a number or a list, invalid
                else:
                    print_error("%s invalid setting for [sensors][current][sensor_scales]" % name +
                                "\n\tmust be a number or a list of numbers")
                    return Status.error
                # 4b3 check sinefit_rotations
                x = current_config["sinefit_rotations"]
                #... must be a list of numbers
                if(not(type(x) is list)):
                    print_error("%s invalid setting for [sensors][current][sinefit_rotations]" % name +
                                "\n\tmust be a list of rotation values in degrees")
                    return Status.error
                #... must be a list of numbers
                for val in x:
                    if(not(type(val) is int) and not(type(val) is float)):
                        print_error("%s invalid value in [sensors][current][sinefit_rotations]" % name +
                                    "\n\tmust be a list of rotations in degrees")
                        return Status.error
                #... length must match number of phases
                if(len(x) != phases):
                    print_error("%s invalid setting for [sensors][current][sinefit_rotations]," % name +
                                "\n\tthe length must match the number of phases")
                    return Status.error
                #... warn if it looks like rotations are in radians
                all_small = True
                for val in x:
                    if(abs(val) > abs(np.pi * 2)):
                        all_small = False
                #... if all rotations are 0 do not raise this warning
                if(all(x)==0):
                    all_small = False
                if(all_small):
                    print_warning("%s settings for [sensors][current][sinefit_rotations] should" % name +
                                  "\n\tbe in degrees, current values look like radians")
                    my_status = min(my_status, Status.warn)
            except KeyError as e:
                print_error(
                    "%s error in [sensors][current]: missing [%s]" % (name, e))
                return Status.error
        except KeyError as e:
            print_error("%s error in [sensors]: missing [%s]" % (name, e))
            return Status.error
        # 5 check streams
        stream_config = config["streams"]
        if(not(type(stream_config) is dict)):
            print_error("%s invalid [streams] settings" % name)
            return Status.error
        try:
            r = min(Status.ok, check_stream_config(
                name, "sinefit", stream_config["sinefit"]))
            r = min(r, check_stream_config(name, "iv", stream_config["iv"]))
            r = min(r, check_stream_config(name, "prep",
                                           stream_config["prep"], allow_no_keep=False))
            # add a sensor stream with no keep
            stream_config["sensor"] = {"keep": '1w',
                                       "decimate": True}
            if(r != Status.ok):
                return Status.error  # check_stream_config prints the error message
            my_status = min(my_status, r)
        except KeyError as e:
            print_error("%s error in [streams]: missing [%s]" % (name, e))
    except KeyError as e:
        print_error("%s missing configuration [%s]" % (name, e))
        return Status.error

    return my_status
