
import logging
import numpy as np
import argparse


from nilm.meter import meter
from joule import LocalPipe, CompositeModule
from joule.utilities import yesno
from .reconstructor import Reconstructor
from .sinefit import Sinefit
from .prep import Prep

ARGS_DESC = """
**view me at http://docs.wattsworth.net/modules**

---
:name:
  NILM RawToPrep
:author:
  John Donnal, James Paris
:license:
  Closed
:url:
  http://git.wattsworth.net/wattsworth/nilm.git
:description:
  run the full filter stack
:usage:
  This is a composite module that runs nilm-reconstructor, nilm-sinefit,
  and nilm-prep. It uses the settings in meters.yml to
  initialize the modules. In most use cases this is preferred over
  running the modules individually

  Sinefit is configured with a frequency of 60Hz +/- 10Hz and min_amp of 10
 
  | Arguments     | Default | Description
  |---------------|---------|---------------------------------
  |``meter``    | --  | meter name in meters.yml
  |``config-file``  | /opt/configs/meters.yml  | location of meters.yml
  |``calibration-file``   | --  | (optional) override default calibration file
  |``calibration-directory``   | /opt/configs/meters  | location of calibration files
  |``merge-prep``    | yes  | [yes|no] single or per-phase prep streams
  |``polar`` | no | [yes|no] polar or cartesian prep output

:inputs:
 
  raw
  : output from nilm-reader module

:outputs:

  zero_crossings
  : ``float32`` with phase, amplitude and offset elements

  
  for merged 3 phase

  prep
  :  ``float32`` [P1A, Q1A, ..., P1B, Q1B, ... P1C, Q1C, ...]

  for unmerged 3 phase
  
  prep-a
  :   ``float32`` [P1, Q1, ... P7, Q7]

  prep-b
  :   ``float32`` [P1, Q1, ... P7, Q7]

  prep-c
  :   ``float32`` [P1, Q1, ... P7, Q7]


:module_config:
    [Main]
    name = NILM RawToPrep
    exec_cmd = nilm-filter-rawtoprep

    [Arguments]
    meter = meter1
    merge = yes
    polar = no

    [Inputs]
    raw = /path/to/raw

    [Outputs]
    zero_crossings = /path/to/zero_crossings
    prep = /path/to/prep
    # for unmerged prep specify prep-a, prep-b, etc.

:stream_configs:
  #zero_crossings#
     [Main]
     name = Zero Crossings
     path = /path/to/zero_crossings
     datatype = float32
     keep = 1w

     [Element1]
     name = Amplitude
     [Element2]
     name = Offset
     [Element3]
     name = Phase
  #prep (merged)#
     [Main]
     name = Prep
     path = /path/to/prep
     datatype = float32
     keep = 1w

     # Phase A
     [Element1]
     name = P1A
     [Element2]
     name = Q1A
     [Element1]
     name = P3A
     [Element2]
     name = Q3A
     [Element1]
     name = P5A
     [Element2]
     name = Q5A
     [Element1]
     name = P7A
     [Element2]
     name = Q7A

     # Phase B
     [Element1]
     name = P1B
     [Element2]
     name = Q1B
     [Element1]
     name = P3B
     [Element2]
     name = Q3B
     [Element1]
     name = P5B
     [Element2]
     name = Q5B
     [Element1]
     name = P7B
     [Element2]
     name = Q7B

     # Phase C
     [Element1]
     name = P1C
     [Element2]
     name = Q1C
     [Element1]
     name = P3C
     [Element2]
     name = Q3C
     [Element1]
     name = P5C
     [Element2]
     name = Q5C
     [Element1]
     name = P7C
     [Element2]
     name = Q7C

  #prep-a (unmerged)#
     [Main]
     name = Prep A
     path = /path/to/prep-a
     datatype = float32
     keep = 1w

     [Element1]
     name = P1
     [Element2]
     name = Q1
     [Element1]
     name = P3
     [Element2]
     name = Q3
     [Element1]
     name = P5
     [Element2]
     name = Q5
     [Element1]
     name = P7
     [Element2]
     name = Q7

---
"""


class RawToPrep(CompositeModule):

    def custom_args(self, parser):
        parser.add_argument("--meter", required=True,
                            help='name from meters.yml (eg meter1)')
        parser.add_argument("--config-file",
                            default="/opt/configs/meters.yml")
        parser.add_argument("--calibration-file",
                            help="override default calibration file")
        parser.add_argument("--calibration-directory",
                            default="/opt/configs/meters")
        parser.add_argument("--merge-prep", type=yesno, default=True,
                            help="store prep in a single stream")
        parser.add_argument("--polar", type=yesno, default=False,
                            help="compute mag,phase instead of PQ")
        parser.add_argument("--frequency", type=float, default=60.0,
                            help="utility line frequency")
        parser.add_argument("--min-voltage", type=float, default=10.0,
                            help="minimum voltage signal to track")
        parser.description = ARGS_DESC

    async def setup(self, parsed_args, inputs, outputs):

        # configure logger
        logger = logging.getLogger()
        logger.setLevel(logging.WARN)
        meters = meter.load_meters(parsed_args.config_file,
                                   parsed_args.calibration_directory)
        my_meter = meters[parsed_args.meter]

        configs = {
            'reconstructor':  self._build_recon_configs(my_meter),
            'sinefit':        self._build_sinefit_configs(my_meter,
                                                          parsed_args.frequency,
                                                          parsed_args.min_voltage),
            'prep':           self._build_prep_configs(my_meter,
                                                       parsed_args.merge_prep,
                                                       parsed_args.polar,
                                                       parsed_args.frequency)
        }
        out_iv = outputs['iv']
        # reconstructor --> sinefit
        r2s_iv = LocalPipe(out_iv.layout, name="r2s_iv", debug=False)
        # reconstructor --> prep
        r2p_iv = LocalPipe(out_iv.layout, name="r2p_iv", debug=False)
        r2s_iv.subscribe(out_iv)
        r2s_iv.subscribe(r2p_iv)

        out_zc = outputs['zero_crossings']
        # sinefit --> prep
        s2p_zc = LocalPipe(out_zc.layout, name="s2p_zc")
        s2p_zc.subscribe(out_zc)

        # create the module tasks
        tasks = []
        my_recon = Reconstructor()
        args = argparse.Namespace(**configs['reconstructor'])
        tasks.append(my_recon.run(args,
                                  {'raw': inputs['raw']},
                                  {'iv': r2s_iv}))

        my_sinefit = Sinefit()
        args = argparse.Namespace(**configs['sinefit'])
        tasks.append(my_sinefit.run(args,
                                    {'iv': r2s_iv},
                                    {'zero_crossings': s2p_zc}))
        
        # --- prep setup: streams may be separate or merged ---
        prep_pipes_out = {}
        for name in outputs:
            if('prep' in name):
                prep_pipes_out[name] = outputs[name]

        my_prep = Prep()
        args = argparse.Namespace(**configs['prep'])
        tasks.append(my_prep.run(args,
                                 {'iv': r2p_iv,
                                  'zero_crossings': s2p_zc},
                                 prep_pipes_out))
        return tasks

    def _build_recon_configs(self, meter):

        def _compute_e_indices(meter):
            if(meter["type"] == "noncontact"):
                return [meter['sensors']['voltage']['sensor_index']]
            else:  # contact meter
                return meter['sensors']['voltage']['sensor_indices']

        def _compute_voltage_matrix(meter):
            if(meter["type"] == "noncontact"):
                return [meter['calibration']['voltage_scale']]
            else:  # contact meter
                voltage_scales = meter['sensors']['voltage']['sensor_scales']
                if(not(type(voltage_scales) is list)):
                    voltage_scales = [voltage_scales]*meter['phases']
                return np.diag(voltage_scales).tolist()
                
        def _compute_current_matrix(meter):
            if(meter["type"] == "noncontact"):
                has_neutral = meter['calibration']['has_neutral']
                if(not has_neutral):
                    return meter['calibration']['full_current_matrix']
                else:
                    return meter['calibration']['current_matrix']
            else:  # contact meter
                current_scales = meter['sensors']['current']['sensor_scales']
                if(not(type(current_scales) is list)):
                    current_scales = [current_scales]*meter['phases']
                return np.diag(current_scales).tolist()

        def _compute_integration_setting(meter):
            if(meter["type"] == "noncontact"):
                return meter['sensors']['voltage']['digitally_integrate']
            else:  # contact meter
                return False
            
        return {
            'm_indices': meter['sensors']['current']['sensor_indices'],
            'e_indices': _compute_e_indices(meter),
            'max_gap': 5,  # seconds
            'current_matrix': _compute_current_matrix(meter),
            'integrate': _compute_integration_setting(meter),
            'voltage_matrix': _compute_voltage_matrix(meter),
        }

    def _build_sinefit_configs(self, meter, freq, amp):
        if(meter["type"] == "contact"):
            v_index = 1 + ['A', 'B', 'C'].index(
                meter["sensors"]["voltage"]["sinefit_phase"])
        else:
            v_index = 1  # noncontact only has one voltage element
        return {
            'v_index': v_index,
            'frequency': freq,
            'min_freq': freq-10,
            'max_freq': freq+10,
            'min_amp': amp,
        }

    def _build_prep_configs(self, meter, merge_output, polar):

        def _compute_rotations(meter):
            if(meter['type'] == 'noncontact'):
                if(meter['calibration']['has_neutral']):
                    return meter['calibration']['sinefit_rotations']
                else:
                    return meter['calibration']['full_sinefit_rotations']
            else:  # contact meter
                rotations = meter['sensors']['current']['sinefit_rotations']
                # convert to radians
                return [x*(2*np.pi)/360.0 for x in rotations]

        def _compute_sampling_frequency(meter):
            if(meter['type'] == 'noncontact'):
                return 3000
            else:  # contact meter
                return 8000

        def _compute_current_indices(meter):
            if(meter['type'] == 'noncontact'):
                # [ts, V, --> I1, I2, I3]
                return np.arange(2, 2 + meter['phases']).tolist()
            else:  # contact meter
                # [ts, V1, V2, V3, --> I1, I2, I3]
                return np.arange(1 + meter['phases'], 1 + 2 * meter['phases']).tolist()

        def _compute_scale_factor(meter):
            line_rms = meter['sensors']['voltage']['nominal_rms_voltage']
            return line_rms / np.sqrt(2)  # convert amps to rms

        def _compute_nshift(meter):
            nshift = meter['streams']['prep']['nshift']
            return nshift

        def _compute_goertzel(meter):
            goertzel = meter['streams']['prep']['goertzel']
            return goertzel
            
        return {
            'nshift': _compute_nshift(meter),
            'nharm': 4,
            'current_indices': _compute_current_indices(meter),
            'rotations': _compute_rotations(meter),
            'scale_factor': _compute_scale_factor(meter),
            'merge': merge_output,
            'polar': polar,
            'samp_freq': _compute_sampling_frequency(meter),
            'goertzel': _compute_goertzel(meter)
        }

    
def main():
    filter = RawToPrep()
    filter.start()

    
if __name__ == "__main__":
    main()


