import unittest
import asyncio
import argparse
import os
import numpy as np
from unittest import mock

from joule import LocalPipe
from nilm.filters import rawtoprep
from nilm.filters import prep, sinefit, reconstructor

CONFIG_PATH = os.path.join(os.path.dirname(
    __file__), 'rawtoprep_files', 'meters.yml')
CAL_DIR_PATH = os.path.join(os.path.dirname(
    __file__), 'rawtoprep_files', 'meters')


class MockReconstructor(reconstructor.Reconstructor):
    async def run(self, parsed_args, inputs, outputs):
        raw = inputs['raw']
        iv = outputs['iv']
        raw_sarray = await raw.read()
        raw.consume(len(raw_sarray))
        # print(raw_sarray)
        raw_sarray["data"] += 1
        # print(raw_sarray)
        await iv.write(raw_sarray)


class MockSinefit(sinefit.Sinefit):
    async def run(self, parsed_args, inputs, outputs):
        iv = inputs['iv']
        zc = outputs['zero_crossings']
        iv_sarray = await iv.read()
        iv.consume(len(iv_sarray))
        iv_sarray["data"] += 1
        # print(iv_sarray)
        await zc.write(iv_sarray)


class MockPrep(prep.Prep):
    async def run(self, parsed_args, inputs, outputs):
        zc = inputs['zero_crossings']
        iv = inputs['iv']
        prep = outputs['prep']
        zc_sarray = await zc.read()
        iv_sarray = await iv.read()
        zc.consume(len(zc_sarray))
        iv.consume(len(iv_sarray))
        iv_sarray["data"][0] += zc_sarray["data"][0]
        # print(iv_sarray)
        await prep.write(iv_sarray)


class TestRawToPrep(unittest.TestCase):
    def setUp(self):
        self.loop = asyncio.new_event_loop()
        self.loop.set_debug=True
        asyncio.set_event_loop(self.loop)

    def tearDown(self):
        self.loop.close()

    @mock.patch("nilm.filters.rawtoprep.Prep", new=MockPrep)
    @mock.patch("nilm.filters.rawtoprep.Sinefit", new=MockSinefit)
    @mock.patch("nilm.filters.rawtoprep.Reconstructor", new=MockReconstructor)
    def test_configures_pipes_to_filters(self):
        # input
        npipe_raw = LocalPipe(layout="uint8_1", name="raw")
        npipe_raw.write_nowait(np.array([[1.0, 0]]))
        # outputs
        npipe_zc = LocalPipe(layout="uint8_1", name="zc")
        npipe_iv = LocalPipe(layout="uint8_1", name="iv")
        npipe_prep = LocalPipe(layout="uint8_1", name="prep")
        args = argparse.Namespace(meter="meter1",
                                  config_file=CONFIG_PATH,
                                  calibration_directory=CAL_DIR_PATH,
                                  merge_prep=True,
                                  polar=False,
                                  frequency=60,
                                  min_voltage=10)
        my_filter = rawtoprep.RawToPrep()

        async def runner():
            tasks = await my_filter.setup(
                args,
                {"raw": npipe_raw},
                {"iv": npipe_iv,
                 "zero_crossings": npipe_zc,
                 "prep": npipe_prep}
            )
            await asyncio.gather(*tasks)
            await npipe_iv.close()
            await npipe_zc.close()
            await npipe_prep.close()
            await npipe_raw.close()
        loop = asyncio.get_event_loop()
        loop.run_until_complete(runner())
        iv_result = npipe_iv.read_nowait(flatten=True).tolist()
        zc_result = npipe_zc.read_nowait(flatten=True).tolist()
        prep_result = npipe_prep.read_nowait(flatten=True).tolist()
        np.testing.assert_array_equal(np.array([[1.0, 1]]), iv_result)
        np.testing.assert_array_equal(np.array([[1.0, 2]]), zc_result)
        np.testing.assert_array_equal(np.array([[1.0, 3]]), prep_result)
