import sys
import numpy as np
from scipy.fft import fft
from joule.utilities import (timestamp_to_human,
                              seconds_to_timestamp)


class SuppressibleWarning(object):

    def __init__(self, maxcount=10, maxsuppress=100):
        self.maxcount = maxcount
        self.maxsuppress = maxsuppress
        self.count = 0
        self.last_msg = ""

    def _write(self, sec, msg):
        if sec:
            now = timestamp_to_human(seconds_to_timestamp(sec)) + ": "
        else:
            now = ""
        sys.stderr.write(now + msg)

    def warn(self, msg, seconds=None):
        self.count += 1
        if self.count <= self.maxcount:
            self._write(seconds, msg)
        if (self.count - self.maxcount) >= self.maxsuppress:
            self.reset(seconds)

    def reset(self, seconds=None):
        if self.count > self.maxcount:
            self._write(seconds, "(%d warnings suppressed)\n" %
                        (self.count - self.maxcount))
        self.count = 0

_t = None
def sfit4(data, fs,num_iters=7):
    global _t
    """(A, f0, phi, C) = sfit4(data, fs)

    Compute 4-parameter (unknown-frequency) least-squares fit to
    sine-wave data, according to IEEE Std 1241-2010 Annex B

    Input:
      data  vector of input samples
      fs    sampling rate (Hz)

    Output:
      Parameters [A, f0,  phi, C] to fit the equation
        x[n] = A * sin(f0/fs * 2 * pi * n + phi) + C
      where n is sample number.  Or, as a function of time:
        x(t) = A * sin(f0 * 2 * pi * t + phi) + C

    by Jim Paris
    (Verified to match sfit4.m)
    """
    N = len(data)

    if _t is None or N!=len(_t):
        t = np.linspace(0, (N - 1) / float(fs), N)
        _t = t
    else:
        t = _t # use stored version, don't run linspace again
    # Estimate frequency using FFT (step b)
    Fc = fft(data)
    F = abs(Fc)
    F[0] = 0   # eliminate DC

    # Find pair of spectral lines with largest amplitude:
    # resulting values are in F(i) and F(i+1)
    i = np.argmax(F[0:int(N / 2)] + F[1:int(N / 2 + 1)])

    # Interpolate FFT to get a better result (from Markus [B37])
    try:
        U1 = np.real(Fc[i])
        U2 = np.real(Fc[i + 1])
        V1 = np.imag(Fc[i])
        V2 = np.imag(Fc[i + 1])
        n = 2 * np.pi / N
        ni1 = n * i
        ni2 = n * (i + 1)
        K = ((V2 - V1) * np.sin(ni1) + (U2 - U1) * np.cos(ni1)) / (U2 - U1)
        Z1 = V1 * (K - np.cos(ni1)) / np.sin(ni1) + U1
        Z2 = V2 * (K - np.cos(ni2)) / np.sin(ni2) + U2
        i = np.arccos((Z2 * np.cos(ni2) - Z1 * np.cos(ni1)) / (Z2 - Z1)) / n
    except Exception:
        # Just go with the biggest FFT peak
        i = np.argmax(F[0:int(N / 2)])

    # Convert to Hz
    f0 = i * float(fs) / N

    # Fit it.  We'll catch exceptions here and just returns zeros
    # if something fails with the least squares fit, etc.
    try:
        # first guess for A0, B0 using 3-parameter fit (step c)
        s = np.zeros(3)
        w = 2 * np.pi * f0

        # Now iterate 7 times (step b, plus 6 iterations of step i)
        for idx in range(num_iters):
            D = np.c_[np.cos(w * t), np.sin(w * t), np.ones(N),
                      -s[0] * t * np.sin(w * t) + s[1] * t * np.cos(w * t)]  # eqn B.16
            s = np.linalg.lstsq(D, data,rcond=-1)[0]  # eqn B.18
            w = w + s[3]  # update frequency estimate

        # Extract results
        A = np.sqrt(s[0] * s[0] + s[1] * s[1])  # eqn B.21
        f0 = w / (2 * np.pi)
        # eqn B.22 (flipped for sin instead of cos)
        phi = np.arctan2(s[0], s[1])
        C = s[2]
        return (A, f0, phi, C)
    except Exception as e:
        # something broke down; just return zeros
        return (0, 0, 0, 0)


def peak_detect(data, delta=0.1):
    """Simple min/max peak detection algorithm, taken from my code
    in the disagg.m from the 10-8-5 paper.

    Returns an array of peaks: each peak is a tuple
      (n, p, is_max)
    where n is the row number in 'data', and p is 'data[n]',
    and is_max is True if this is a maximum, False if it's a minimum,
    """
    peaks = []
    cur_min = (None, np.inf)
    cur_max = (None, -np.inf)
    lookformax = False
    for (n, p) in enumerate(data):
        if p > cur_max[1]:
            cur_max = (n, p)
        if p < cur_min[1]:
            cur_min = (n, p)
        if lookformax:
            if p < (cur_max[1] - delta):
                peaks.append((cur_max[0], cur_max[1], True))
                cur_min = (n, p)
                lookformax = False
        else:
            if p > (cur_min[1] + delta):
                peaks.append((cur_min[0], cur_min[1], False))
                cur_max = (n, p)
                lookformax = True
    return peaks

def goertzel(x,f_samp,f_vector):
    N = len(x)
    delta_f = f_samp/N
    ind_vector = f_vector/delta_f

    DFT = np.zeros(len(ind_vector),dtype = np.complex)

    for ind, k in enumerate(ind_vector):

        # precalculation of constants 
        A = 2*np.pi*k/N
        B = 2*np.cos(A)
        C = np.exp(-1j*A)
        D = np.exp(-1j*A*(N-1))

        # State variables
        s0 = 0
        s1 = 0
        s2 = 0

        # Main loop
        # number of iterations is one less than the length of the signal
        for n in range(0, N-1):
            # new state
            s0 = x[n] + B*s1 -s2
            # shifting the state variables
            s2 = s1
            s1 = s0

        # Finalizing calculations
        s0 = x[-1] + B*s1 - s2
        y = s0 - s1*C               # resulting complex coefficient
        y = y*D                     # constant substituting the iteration N-1, and correcting the phase at the same time
     
        DFT[ind] = 2.0*y/N

    return DFT


class FilterError(Exception):
    pass
