#!/usr/bin/python

"""
Check the meters.yml configuration for errors
and generate warnings for disk usage and sensor
status

John Donnal 2016
"""

import yaml
import re

from nilm.meter.utils import Status, print_error, MeterConfigurationError
from nilm.meter.connectivity import check_meter_connections
from nilm.meter.storage import check_disk_usage
from nilm.meter import contact
from nilm.meter import noncontact
from nilm.meter import calibration


METER_DEV = "/dev/nilm/%s-data"


def load_meters(meter_file="/opt/configs/meters.yml",
                calibration_dir="/opt/configs/meters",
                verbose=False,
                check_connectivity=False):

    meters = _load_meter_file(meter_file)
    if(meters == Status.error):
        raise MeterConfigurationError
    
    if(_validate_meters(meters, verbose) == Status.error):
        raise MeterConfigurationError

    for meter in meters.values():
        # add the TTY location to the configs
        if(meter['type'] == 'noncontact'):
            meter['tty'] = METER_DEV % meter['serial_number']
        calibration.load_calibration(calibration_dir, meter)
    if(check_connectivity):
        check_meter_connections(meters)
    return meters


def _load_meter_file(meter_file):
    ############
    # load the meters file
    try:
        with open(meter_file, 'r') as f:
            meters = yaml.safe_load(f)
    except FileNotFoundError:
        raise MeterConfigurationError(
            "can't load meters file at [%s], is it missing?" % meter_file)
    except Exception as e:
        raise MeterConfigurationError(str(e)) from e

    # add the meter name as an element of the configuration
    if(meters is None):
        raise MeterConfigurationError("[/opt/configs/meters.yml] is empty")

    if(type(meters) != dict):
        raise MeterConfigurationError("[/opt/configs/meters.yml] has invalid syntax")
    for (name, meter) in meters.items():
        meter['name'] = name
        # add a path attribute if it does not exist
        if 'path' not in meter:
            meter['path'] = "/"+meter['name']
    ##############
    # check for duplicate meter names,
    #    pyyaml has a bug that allows duplicates :<
    meter_names = []
    with open(meter_file, 'r') as f:
        for line in f:
            x = re.match("^(meter\d+)\s*:$", line)
            if(x is not None):
                meter_names.append(x.group(1))
    if(len(set(meter_names)) != len(meter_names)):
        print(
            "each meter must have a unique name (meter1, meter2, ...)")
        return Status.error
    return meters


def _validate_meters(meters, verbose):

    # run through a series of checks
    my_status = Status.ok
    r = _check_syntax(meters)
    if r == Status.error:
        return r
    else:
        my_status = min(r, my_status)
    if r == Status.error:
        return r
    else:
        my_status = min(r, my_status)
    # TOOD: this should be done by jouled
    #    r = check_disk_usage(meters, verbose)
    r = Status.ok
    if r == Status.error:
        return r
    else:
        my_status = min(r, my_status)
    # no errors, return the worst status code
    return my_status


# Check if the syntax in the configuration file is valid
#  Error for invalid configs
#  Warning for non-standard settings
def _check_syntax(settings):
    my_status = Status.ok
    # 0 Make sure there is something in the configuration
    if(settings is None):
        print_error("meters.yml is empty", "software#software_configuration")
        return Status.error
    for meter in settings:
        # 1 Make sure meter has a valid name
        if(re.match("^meter\d+$", meter) is None):
            print_error("invalid name [%s], use unique names [meter1,meter2,...]" % meter,
                        "software#software_configuration")
            return Status.error
        meter_config = settings[meter]
        # 2 Make sure meter type is present and valid
        try:
            if(meter_config["type"] == "noncontact"):
                # 3a Check non-contact meter settings
                r = noncontact.check_syntax(meter, meter_config)
            elif(meter_config["type"] == "contact"):
                # 3b Check contact meter settings
                r = contact.check_syntax(meter, meter_config)
            else:
                print_error("%s invalid type [%s], must be [contact | noncontact]" % (meter,
                                                                                      meter_config["type"]), "software#software_configuration")
                return Status.error
            if(r == Status.error):
                return r
            else:
                my_status = min(r, my_status)
        except KeyError:
            print_error("%s missing [type], must be [contact | noncontact]" % meter,
                        "software#software_configuration")

    return my_status
