#!/usr/bin/python

# To release a new version, tag it:
#   git tag -a nilmdb-1.1 -m "Version 1.1"
#   git push --tags
# Then just package it up:
#   python setup.py sdist

import traceback
import sys
import os

try:
    from setuptools import setup, find_packages
    from distutils.extension import Extension
    import distutils.version
except ImportError:
    traceback.print_exc()
    print "Please install the prerequisites listed in README.txt"
    sys.exit(1)

# Versioneer manages version numbers from git tags.
# https://github.com/warner/python-versioneer
import versioneer
versioneer.versionfile_source = 'nilmdb/_version.py'
versioneer.versionfile_build = 'nilmdb/_version.py'
versioneer.tag_prefix = 'nilmdb-'
versioneer.parentdir_prefix = 'nilmdb-'

# Hack to workaround logging/multiprocessing issue:
# https://groups.google.com/d/msg/nose-users/fnJ-kAUbYHQ/_UsLN786ygcJ
try: import multiprocessing
except Exception: pass

# Use Cython if it's new enough, otherwise use preexisting C files.
cython_modules = [ 'nilmdb.server.interval',
                   'nilmdb.server.rbtree' ]
try:
    import Cython
    from Cython.Build import cythonize
    if (distutils.version.LooseVersion(Cython.__version__) <
        distutils.version.LooseVersion("0.16")):
        print "Cython version", Cython.__version__, "is too old; not using it."
        raise ImportError()
    use_cython = True
except ImportError:
    use_cython = False

ext_modules = [ Extension('nilmdb.server.rocket', ['nilmdb/server/rocket.c' ]) ]
for modulename in cython_modules:
    filename = modulename.replace('.','/')
    if use_cython:
        ext_modules.extend(cythonize(filename + ".pyx"))
    else:
        cfile = filename + ".c"
        if not os.path.exists(cfile):
            raise Exception("Missing source file " + cfile + ".  "
                            "Try installing cython >= 0.16.")
        ext_modules.append(Extension(modulename, [ cfile ]))

# We need a MANIFEST.in.  Generate it here rather than polluting the
# repository with yet another setup-related file.
with open("MANIFEST.in", "w") as m:
    m.write("""
# Root
include README.txt
include setup.cfg
include setup.py
include versioneer.py
include Makefile
include .coveragerc
include .pylintrc

# Cython files -- include source.
recursive-include nilmdb/server *.pyx *.pyxdep *.pxd

# Tests
recursive-include tests *.py
recursive-include tests/data *
include tests/test.order

# Docs
recursive-include docs Makefile *.md

# Extras
recursive-include extras *
""")

# Run setup
setup(name='nilmdb',
      version = versioneer.get_version(),
      cmdclass = versioneer.get_cmdclass(),
      url = 'https://git.jim.sh/jim/lees/nilmdb.git',
      author = 'Jim Paris',
      description = "NILM Database",
      long_description = "NILM Database",
      license = "Proprietary",
      author_email = 'jim@jtan.com',
      tests_require = [ 'nose',
                        'coverage',
                        'numpy',
                        ],
      setup_requires = [ 'setuptools',
                         ],
      install_requires = [ 'decorator',
                           'cherrypy >= 3.2',
                           'simplejson',
                           'python-dateutil',
                           'pytz',
                           'psutil >= 0.3.0',
                           'requests >= 1.1.0',
                           'progressbar >= 2.2',
                           ],
      packages = [ 'nilmdb',
                   'nilmdb.utils',
                   'nilmdb.utils.datetime_tz',
                   'nilmdb.server',
                   'nilmdb.client',
                   'nilmdb.cmdline',
                   'nilmdb.scripts',
                   'nilmdb.fsck',
                   ],
      entry_points = {
          'console_scripts': [
              'nilmtool = nilmdb.scripts.nilmtool:main',
              'nilmdb-server = nilmdb.scripts.nilmdb_server:main',
              'nilmdb-fsck = nilmdb.scripts.nilmdb_fsck:main',
              ],
          },
      ext_modules = ext_modules,
      zip_safe = False,
      )
