#!/usr/bin/python

# Copy streams from a source path into a different 
# destination path
# eg: /data/prep-a -> /ship/prep-a
# Works as a wildcard copy

import nilmtools.filter
import nilmtools.copy_one
import nilmdb.client
import argparse
import fnmatch
import re
import pdb

def main(argv = None):
    f = nilmtools.filter.Filter()
    # Reuse filter's parser, since it handles most options we need.
    parser = f.setup_parser(description = """\
    Copy all streams matching the given wildcard from one host 
    to another renamed to the destination wildcard. Used to archive
    streams collected on another NILM

    Example: %(prog)s -u http://localhost:8080 -U http://localhost/nilmdb /data/* /sharon/*
    """, skip_paths = True)
    parser.add_argument('-x', '--nometa', action='store_true',
                        help="Don't copy or check metadata")
    parser.add_argument("src_path", action="store", 
                        help='Wildcard source path to copy')
    parser.add_argument("dest_path", action="store", 
                        help='Wildcard destination')
    args = parser.parse_args(argv)

    # Verify arguments
    if args.dest_url is None:
        parser.error("must provide both source and destination URL")
    client_src = nilmdb.client.Client(args.url)
    client_dest = nilmdb.client.Client(args.dest_url)
    print "Source URL:", client_src.geturl()
    print "  Dest URL:", client_dest.geturl()
    print " %s --> %s "%(args.src_path,args.dest_path)

    # Find matching streams
    sources = []
    sources.extend([s for s in client_src.stream_list(extended = True)
                    if fnmatch.fnmatch(s[0], args.src_path)
                    and s not in sources])
    # Figure out the source and dest groups /data/* -> data
    m = re.match("/(\w+)/\*",args.src_path)
    src_group = m.group(1)
    m = re.match("/(\w+)/\*",args.dest_path)
    dest_group = m.group(1)
    # Create destination streams if they don't exist
    dests = []
    for stream in sources:
        src = nilmtools.filter.StreamInfo(client_src.geturl(), stream)
        # change src.path to the dest_path
        dest_path = re.sub("/\w+/*","/%s/"%dest_group,src.path)
        dest = nilmtools.filter.get_stream_info(client_dest, dest_path)
        if not dest:
            print "Creating destination stream", dest_path
            client_dest.stream_create(dest_path, src.layout)

    # Copy them all by running the "copy" tool as if it were
    # invoked from the command line.
    for stream in sources:
        new_argv = ["--url", client_src.geturl(),
                     "--dest-url", client_dest.geturl() ]
        if args.start:
            new_argv.extend(["--start", "@" + repr(args.start)])
        if args.end:
            new_argv.extend(["--end", "@" + repr(args.end)])
        if args.dry_run:
            new_argv.extend(["--dry-run"])
        if args.nometa:
            new_argv.extend(["--nometa"])
        if args.force_metadata:
            new_argv.extend(["--force-metadata"])
        dest_path = re.sub("/\w+/*","/%s/"%dest_group,stream[0])
        new_argv.extend([stream[0], dest_path])
        try:
            nilmtools.copy_one.main(new_argv)
        except SystemExit as e:
            # Ignore SystemExit which could be raised on --dry-run
            if e.code != 0:
                raise

if __name__=="__main__":
    main()
